<?php

namespace Drupal\access_unpublished_group\Access;

use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\group\Entity\GroupInterface;
use Symfony\Component\Routing\Route;

/**
 * Access check for group moderation tab which supports access_unpublished.
 *
 * This decorator wraps the Group module's group_latest_revision access check
 * to add Access Unpublished token support for the /group/{group}/latest route.
 */
class GroupLatestRevisionCheck implements AccessInterface {

  /**
   * The decorated access check.
   *
   * @var \Drupal\Core\Routing\Access\AccessInterface
   */
  protected AccessInterface $accessCheck;

  /**
   * Constructs a GroupLatestRevisionCheck object.
   *
   * @param \Drupal\Core\Routing\Access\AccessInterface $access_check
   *   The group latest revision access check to decorate.
   */
  public function __construct(AccessInterface $access_check) {
    $this->accessCheck = $access_check;
  }

  /**
   * Checks access to the group latest revision route.
   *
   * Combines the Group module's access check with Access Unpublished token
   * validation to allow token-based access to draft group revisions.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The parametrized route.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The current user account.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, RouteMatchInterface $route_match, AccountInterface $account): AccessResultInterface {
    $access = $this->accessCheck->access($route, $route_match, $account);

    $group = $route_match->getParameter('group');
    if ($group instanceof GroupInterface) {
      $access = $access->orIf(access_unpublished_entity_access($group, 'view', $account));
    }

    return $access;
  }

}
