<?php

namespace Drupal\access_unpublished_group;

use Drupal\access_unpublished_group\Access\GroupLatestRevisionCheck;
use Drupal\access_unpublished_group\Plugin\Group\RelationHandler\AccessUnpublishedGroupAccessControl;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\DependencyInjection\ServiceProviderBase;
use Drupal\Core\Plugin\Discovery\AttributeDiscoveryWithAnnotations;
use Drupal\group\Plugin\Attribute\GroupRelationType;
use Drupal\group\Plugin\Group\Relation\GroupRelationTypeInterface;
use Symfony\Component\DependencyInjection\Reference;

/**
 * Decorates the group.relation_handler.access_control.* services.
 *
 * Also decorates the access_check.group_latest_revision service to support
 * Access Unpublished tokens on the /group/{group}/latest route.
 */
class AccessUnpublishedGroupServiceProvider extends ServiceProviderBase {

  /**
   * {@inheritdoc}
   *
   * This implementation alters existing services through use of
   * `ServiceProvider::alter()` method, see related Drupal API documentation:
   * https://www.drupal.org/docs/drupal-apis/services-and-dependency-injection/altering-existing-services-providing-dynamic-services.
   *
   * The services being altered are first defined (altered) in Group's contrib
   * module implementation of `ServiceProvider::alter()`, implemented in
   * `\Drupal\group\GroupServiceProvider`.
   *
   * If Group contrib module is updated and the Access Unpublished functionality
   * stops working for Group content, this is the first place to look into as
   * the `\Drupal\group\GroupServiceProvider::alter()` method might have been
   * changed. If so, the changes must be reflected in this method.
   *
   * @see \Drupal\group\GroupServiceProvider::alter()
   * @see \Drupal\access_unpublished\AccessUnpublishedServiceProvider::alter()
   * @see https://www.drupal.org/docs/drupal-apis/services-and-dependency-injection/altering-existing-services-providing-dynamic-services
   */
  public function alter(ContainerBuilder $container) {
    // Decorate the group latest revision access check to support Access
    // Unpublished tokens on /group/{group}/latest routes.
    $this->decorateGroupLatestRevisionAccessCheck($container);

    // Get list of all modules (custom and contrib).
    $modules = $container->getParameter('container.modules');
    // Prepare discovery.
    $discovery = $this->prepareGroupRelationsDiscovery($container);

    // Iterate through all group relations and decorate their access_control
    // handler.
    foreach ($discovery->getDefinitions() as $group_relation_type_id => $group_relation_type) {
      assert($group_relation_type instanceof GroupRelationTypeInterface);
      // Skip plugins whose provider is not installed.
      if (!isset($modules[$group_relation_type->getProvider()])) {
        continue;
      }

      $existing_service_name = "group.relation_handler.access_control.$group_relation_type_id";
      $new_service_name = "access_unpublished_group.relation_handler.access_control.$group_relation_type_id";

      // Decorate the service if it exists.
      if ($container->hasDefinition($existing_service_name)) {
        $container->register($new_service_name, AccessUnpublishedGroupAccessControl::class)
          ->setDecoratedService($existing_service_name)
          ->addArgument(new Reference("$new_service_name.inner"))
          ->setPublic(TRUE)
          ->setShared(FALSE);
      }
    }
  }

  /**
   * Decorates the group latest revision access check service.
   *
   * The Group module replaces the standard _content_moderation_latest_version
   * access requirement with its own _group_latest_revision requirement. This
   * bypasses Access Unpublished's decorator on access_check.latest_revision.
   *
   * This method decorates access_check.group_latest_revision to add token
   * support for the /group/{group}/latest route.
   *
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The container builder.
   *
   * @see \Drupal\group\Entity\Routing\GroupLatestRevisionRouteSubscriber
   * @see \Drupal\access_unpublished\AccessUnpublishedServiceProvider::alter()
   */
  protected function decorateGroupLatestRevisionAccessCheck(ContainerBuilder $container): void {
    if ($container->hasDefinition('access_check.group_latest_revision')) {
      $container->register('access_unpublished_group.access_check.group_latest_revision', GroupLatestRevisionCheck::class)
        ->setDecoratedService('access_check.group_latest_revision')
        ->addArgument(new Reference('access_unpublished_group.access_check.group_latest_revision.inner'));
    }
  }

  /**
   * Prepare group relations discovery.
   *
   * The discovery changed between Group contrib versions 2.2.0 and 2.3.1 and
   * as a consequence it had to be copied from
   * `\Drupal\group\GroupServiceProvider::alter()`.
   *
   * @param \Drupal\Core\DependencyInjection\ContainerBuilder $container
   *   The ContainerBuilder whose service definitions can be altered.
   *
   * @return \Drupal\Core\Plugin\Discovery\AnnotatedClassDiscovery
   *   The discovery of group relations.
   *
   * @see \Drupal\group\GroupServiceProvider::alter()
   */
  protected function prepareGroupRelationsDiscovery(ContainerBuilder $container) {
    return new AttributeDiscoveryWithAnnotations(
      'Plugin/Group/Relation',
      $container->get('container.namespaces'),
      GroupRelationType::class,
      'Drupal\group\Annotation\GroupRelationType',
      []
    );
  }

}
