<?php

namespace Drupal\Tests\access_unpublished_group\Functional;

use Drupal\access_unpublished\Entity\AccessToken;
use Drupal\group\PermissionScopeInterface;
use Drupal\Tests\group\Functional\GroupBrowserTestBase;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;
use Drupal\user\RoleInterface;

/**
 * Tests the content access in a group.
 */
class ContentInGroupAccessTest extends GroupBrowserTestBase {

  use ContentTypeCreationTrait;
  use NodeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'gnode',
    'group_test_config',
    'access_unpublished_group',
  ];

  /**
   * Group type.
   *
   * @var \Drupal\group\Entity\GroupTypeInterface
   */
  protected $groupType;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Get default group type from group_test_config.
    $this->groupType = $this->entityTypeManager->getStorage('group_type')->load('default');

    // Add the page content type to the group type.
    $this->createContentType(['type' => 'page']);
    /** @var \Drupal\group\Entity\Storage\GroupRelationshipTypeStorageInterface $group_relationship_type_storage */
    $group_relationship_type_storage = $this->entityTypeManager->getStorage('group_content_type');
    $group_relationship_type_storage->createFromPlugin($this->groupType, 'group_node:page')->save();
    // Clearing cache because the service provider is only configured on cache
    // clear after the relationship above is configured.
    drupal_flush_all_caches();

    user_role_grant_permissions(RoleInterface::ANONYMOUS_ID, [
      'access content',
      'access_unpublished node page',
    ]);

    $this->createGroupRole([
      'group_type' => $this->groupType->id(),
      'scope' => PermissionScopeInterface::OUTSIDER_ID,
      'global_role' => RoleInterface::ANONYMOUS_ID,
      'permissions' => ['access_unpublished_group_group_node:page'],
    ]);
  }

  /**
   * Checks group entity access before and after token creation.
   */
  public function testGroupEntityAccessWithValidToken() {
    $assert_session = $this->assertSession();

    // Create an unpublished entity.
    $entity = $this->createNode(['type' => 'page', 'status' => FALSE]);

    // Create a group and add entity to the group.
    $group = $this->createGroup(['type' => $this->groupType->id()]);
    $group->addRelationship($entity, 'group_node:page');

    // Create tokens for the entity.
    $requestTime = \Drupal::time()->getRequestTime();
    $validToken = AccessToken::create([
      'entity_type' => 'node',
      'entity_id' => $entity->id(),
      'value' => 'iAmValid',
      'expire' => $requestTime + 100,
    ]);
    $validToken->save();

    // Verify that unpublished entity is not accessible.
    $this->drupalGet($entity->toUrl('canonical'));
    $assert_session->statusCodeEquals(403);
    // Verify that entity is accessible, but only with the correct hash.
    $this->drupalGet($entity->toUrl('canonical'), ['query' => ['auHash' => 'iAmValid']]);
    $assert_session->statusCodeEquals(200);
    $this->drupalGet($entity->toUrl('canonical'), ['query' => ['auHash' => 123456]]);
    $assert_session->statusCodeEquals(403);
    $this->drupalGet($entity->toUrl());
    $assert_session->statusCodeEquals(403);

    // Delete the token.
    $validToken->delete();

    // Verify that the entity is not accessible.
    $this->drupalGet($entity->toUrl('canonical'), ['query' => ['auHash' => 'iAmValid']]);
    $assert_session->statusCodeEquals(403);
  }

  /**
   * Checks group entity access with expired token.
   */
  public function testAccessWithExpiredToken() {
    $assert_session = $this->assertSession();

    // Create an unpublished entity.
    $entity = $this->createNode(['type' => 'page', 'status' => FALSE]);

    // Create a group and add entity to the group.
    $group = $this->createGroup(['type' => $this->groupType->id()]);
    $group->addRelationship($entity, 'group_node:page');

    // Create a token for the entity.
    $token = AccessToken::create([
      'entity_type' => 'node',
      'entity_id' => $entity->id(),
      'value' => '12345',
      'expire' => \Drupal::time()->getRequestTime() - 100,
    ]);
    $token->save();

    $this->drupalGet($entity->toUrl('canonical'), ['query' => ['auHash' => 12345]]);
    $assert_session->statusCodeEquals(403);
  }

  /**
   * Checks group entity access modified header.
   */
  public function testAccessModifiedHeader() {
    $assert_session = $this->assertSession();

    // Create an unpublished entity.
    $entity = $this->createNode(['type' => 'page', 'status' => FALSE]);

    // Create a group and add entity to the group.
    $group = $this->createGroup(['type' => $this->groupType->id()]);
    $group->addRelationship($entity, 'group_node:page');

    // Create a token for the entity.
    $validToken = AccessToken::create([
      'entity_type' => 'node',
      'entity_id' => $entity->id(),
      'value' => 'iAmValid',
      'expire' => -1,
    ]);
    $validToken->save();

    $this->drupalGet($entity->toUrl(), ['query' => ['auHash' => 'iAmValid']]);
    $assert_session->statusCodeEquals(200);
    $assert_session->responseHeaderNotContains('X-Robots-Tag', 'noindex');

    \Drupal::configFactory()->getEditable('access_unpublished.settings')
      ->set('modify_http_headers', ['X-Robots-Tag' => 'noindex'])
      ->save();

    $this->drupalGet($entity->toUrl(), ['query' => ['auHash' => 'iAmValid']]);
    $assert_session->statusCodeEquals(200);
    $assert_session->responseHeaderContains('X-Robots-Tag', 'noindex');
  }

}
