<?php

declare(strict_types=1);

namespace Drupal\Tests\access_unpublished_group\Functional;

use Drupal\Tests\content_moderation\Traits\ContentModerationTestTrait;
use Drupal\Tests\group\Functional\GroupBrowserTestBase;
use Drupal\access_unpublished\Entity\AccessToken;
use Drupal\access_unpublished_group\Access\GroupLatestRevisionCheck;
use Drupal\user\RoleInterface;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the decorator on the group_latest_revision access check.
 */
#[Group('access_unpublished_group')]
#[CoversClass(GroupLatestRevisionCheck::class)]
class GroupLatestRevisionAccessTest extends GroupBrowserTestBase {

  use ContentModerationTestTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'content_moderation',
    'group_test_config',
    'access_unpublished_group',
  ];

  /**
   * The test group.
   *
   * @var \Drupal\group\Entity\GroupInterface
   */
  protected $group;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->createEditorialWorkflow();

    // Add the default group type to the editorial workflow.
    /** @var \Drupal\workflows\WorkflowInterface $workflow */
    $workflow = $this->entityTypeManager->getStorage('workflow')->load('editorial');
    $workflow->getTypePlugin()->addEntityTypeAndBundle('group', 'default');
    $workflow->save();

    // Create and publish a group.
    $this->group = $this->createGroup([
      'type' => 'default',
      'moderation_state' => 'published',
    ]);
    // Create a new revision for the group in draft state.
    $this->group->setNewRevision();
    $this->group->set('moderation_state', 'draft');
    $this->group->save();

    // Allow anonymous users to access unpublished groups with a token.
    user_role_grant_permissions(RoleInterface::ANONYMOUS_ID, [
      'access content',
      'access_unpublished group default',
    ]);
  }

  /**
   * Checks group revision access with and without a valid token.
   */
  public function testGroupLatestRevisionAccess() {
    // Confirm access to the latest revision is denied without a token.
    $latest_revision_url = "/group/{$this->group->id()}/latest";
    $this->drupalGet($latest_revision_url);
    $this->assertSession()->statusCodeEquals(403);

    // Create a token for the group.
    $token_id = 'iAmValidForGroup';
    $request_time = \Drupal::time()->getRequestTime();
    $token = AccessToken::create([
      'entity_type' => 'group',
      'entity_id' => $this->group->id(),
      'value' => $token_id,
      'expire' => $request_time + 100,
    ]);
    $token->save();
    $token_url_options = ['query' => ['auHash' => $token_id]];

    // Confirm access is still denied if not using the token.
    $this->drupalGet($latest_revision_url);
    $this->assertSession()->statusCodeEquals(403);

    // Confirm access granted with a valid token.
    $this->drupalGet($latest_revision_url, $token_url_options);
    $this->assertSession()->statusCodeEquals(200);

    // Confirm access denied with an invalid token.
    $this->drupalGet($latest_revision_url, ['query' => ['auHash' => 'SomethingElse']]);
    $this->assertSession()->statusCodeEquals(403);

    // Confirm that access is denied with an expired token.
    $token->set('expire', $request_time - 100);
    $token->save();
    $this->drupalGet($latest_revision_url, $token_url_options);
    $this->assertSession()->statusCodeEquals(403);

    // Confirm that access is denied with a deleted token.
    $token->delete();
    $this->drupalGet($latest_revision_url, $token_url_options);
    $this->assertSession()->statusCodeEquals(403);
  }

}
