<?php

declare(strict_types=1);

namespace Drupal\accessibility_filters\Plugin\Filter;

use Drupal\accessibility_filters\Utility\EmptyElementTrait;
use Drupal\Component\Utility\Html;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filter\Attribute\Filter;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\Plugin\FilterInterface;

/**
 * Removes empty headings by replacing them with paragraph tags.
 *
 * This filter scans h1–h6 and, when a heading has no visible text content,
 * replaces it with a <p> while preserving original attributes and any children.
 */
#[Filter(
  id: "empty_headings_filter",
  title: new TranslatableMarkup("Empty headings filter"),
  description: new TranslatableMarkup("Replaces empty heading elements (&lt;h1&gt;-&lt;h6&gt;) with &lt;p&gt; tags, preserving attributes and children."),
  type: FilterInterface::TYPE_TRANSFORM_REVERSIBLE,
  weight: 10
)]
final class EmptyHeadingsFilter extends FilterBase {

  use EmptyElementTrait;

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode): FilterProcessResult {
    if (empty($text)) {
      return new FilterProcessResult((string) $text);
    }

    $dom = Html::load($text);
    $xpath = new \DOMXPath($dom);

    $nodeList = $xpath->query('//h1|//h2|//h3|//h4|//h5|//h6');

    if ($nodeList === FALSE || $nodeList->length === 0) {
      return new FilterProcessResult($text);
    }

    // Copy to array to avoid mutating a live NodeList during replacements.
    $headings = [];
    foreach ($nodeList as $el) {
      $headings[] = $el;
    }

    foreach ($headings as $heading) {
      if (!($heading instanceof \DOMElement)) {
        continue;
      }

      if ($this->isElementEmpty($heading)) {
        $this->replaceHeadingWithParagraph($dom, $heading);
      }
    }

    $output = Html::serialize($dom);
    return new FilterProcessResult($output);
  }

  /**
   * Replace a heading with a <p>, preserving attributes and children.
   */
  private function replaceHeadingWithParagraph(\DOMDocument $dom, \DOMElement $heading): void {
    $p = $dom->createElement('p');

    // Preserve all attributes from the heading.
    if ($heading->hasAttributes()) {
      /** @var \DOMAttr $attr */
      foreach ($heading->attributes as $attr) {
        $p->setAttribute($attr->nodeName, $attr->nodeValue ?? '');
      }
    }

    // Move all children to the new <p>.
    while ($heading->firstChild) {
      $p->appendChild($heading->firstChild);
    }

    // Replace node in DOM.
    $heading->parentNode?->replaceChild($p, $heading);
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE) {
    return (string) $this->t('Converts empty headings (H1–H6) into paragraphs to avoid accessibility issues while keeping layout spacing.');
  }

}
