<?php

declare(strict_types=1);

namespace Drupal\accessibility_filters\Plugin\Filter;

use Drupal\accessibility_filters\Utility\EmptyElementTrait;
use Drupal\Component\Utility\Html;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\filter\Attribute\Filter;
use Drupal\filter\FilterProcessResult;
use Drupal\filter\Plugin\FilterBase;
use Drupal\filter\Plugin\FilterInterface;

/**
 * Removes empty anchor tags for accessibility compliance.
 *
 * This filter scans <a> elements and completely removes any that have
 * no visible text content, as they serve no purpose and break accessibility.
 */
#[Filter(
  id: "empty_links_filter",
  title: new TranslatableMarkup("Empty links filter"),
  description: new TranslatableMarkup("Removes empty &lt;a&gt; elements."),
  type: FilterInterface::TYPE_TRANSFORM_REVERSIBLE,
  weight: 10
)]
final class EmptyLinksFilter extends FilterBase {

  use EmptyElementTrait;

  /**
   * {@inheritdoc}
   */
  public function process($text, $langcode): FilterProcessResult {
    if (empty($text)) {
      return new FilterProcessResult((string) $text);
    }

    $dom = Html::load($text);
    $xpath = new \DOMXPath($dom);

    $nodeList = $xpath->query('//a');

    if ($nodeList === FALSE || $nodeList->length === 0) {
      return new FilterProcessResult($text);
    }

    // Copy to array to avoid mutating a live NodeList during removal.
    $links = [];
    foreach ($nodeList as $el) {
      $links[] = $el;
    }

    foreach ($links as $link) {
      if (!($link instanceof \DOMElement)) {
        continue;
      }

      if ($this->isElementEmpty($link)) {
        // Completely remove the empty link from the DOM.
        $link->parentNode?->removeChild($link);
      }
    }

    $output = Html::serialize($dom);
    return new FilterProcessResult($output);
  }

  /**
   * {@inheritdoc}
   */
  public function tips($long = FALSE) {
    return (string) $this->t('Removes empty anchor (<a>) tags to prevent accessibility issues.');
  }

}
