<?php

declare(strict_types=1);

namespace Drupal\accessibility_filters\Utility;

/**
 * Provides shared logic for detecting and manipulating empty DOM elements.
 */
trait EmptyElementTrait {

  /**
   * Determine if a DOM element has any visible text content.
   *
   * Rules:
   * - Strip comments before measuring text.
   * - Treat &nbsp; as whitespace.
   * - Collapses whitespace and trims.
   */
  protected function isElementEmpty(\DOMElement $element): bool {
    // Clone so we can strip comments without touching the live DOM.
    $clone = $element->cloneNode(TRUE);
    \assert($clone instanceof \DOMElement);

    $this->removeComments($clone);

    // Normalize text content.
    $text = $clone->textContent ?? '';
    // Decode HTML entities and replace NBSP with regular spaces.
    $text = html_entity_decode($text, ENT_QUOTES | ENT_HTML5);
    $text = preg_replace('/\x{00A0}/u', ' ', $text) ?? $text;
    // Collapse whitespace and trim.
    $text = trim(preg_replace('/\s+/u', ' ', $text) ?? '');

    return $text === '';
  }

  /**
   * Remove all comment nodes from a subtree (in place).
   */
  protected function removeComments(\DOMNode $node): void {
    for ($child = $node->firstChild; $child !== NULL; $child = $child->nextSibling) {
      if ($child->nodeType === XML_COMMENT_NODE) {
        $node->removeChild($child);
        // Restart from the beginning since we've mutated siblings.
        $this->removeComments($node);
        return;
      }
      if ($child->hasChildNodes()) {
        $this->removeComments($child);
      }
    }
  }

}
