<?php

declare(strict_types=1);

namespace Drupal\Tests\accessibility_filters\Unit;

use Drupal\accessibility_filters\Plugin\Filter\EmptyHeadingsFilter;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

#[CoversClass(EmptyHeadingsFilter::class)]
#[Group('accessibility_filters')]
final class EmptyHeadingsFilterTest extends UnitTestCase {

  /**
   * The filter plugin instance.
   *
   * @var \Drupal\accessibility_filters\Plugin\Filter\EmptyHeadingsFilter
   */
  private EmptyHeadingsFilter $filter;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // The filter has no constructor deps; instantiate directly.
    $this->filter = new EmptyHeadingsFilter([], 'empty_headings_filter', ['provider' => 'accessibility_filters']);
  }

  /**
   * Small normalization to reduce brittle string assertions.
   */
  private function normalize(string $html): string {
    $html = html_entity_decode($html, ENT_QUOTES | ENT_HTML5);
    // NBSP -> space.
    $html = preg_replace('/\x{00A0}/u', ' ', $html) ?? $html;
    // Collapse ws.
    $html = preg_replace('/\s+/', ' ', $html) ?? $html;
    return trim($html);
  }

  /**
   * Assert no empty headings remain in the output.
   */
  private function assertNoEmptyHeadings(string $html): void {
    $this->assertDoesNotMatchRegularExpression(
      '/<h[1-6]>(\s|&nbsp;|<!--.*?-->|<[^>]*>\s*<\/[^>]*>)*<\/h[1-6]>/si',
      $html
    );
  }

  #[DataProvider('providerEmptyHeadingsAreReplacedWithParagraphs')]
  public function testEmptyHeadingsAreReplacedWithParagraphs(string $input, string $expectedFragment): void {
    $result = $this->filter->process($input, 'en');
    $output = $this->normalize($result->getProcessedText());

    $this->assertIsString($output);
    $this->assertStringContainsString($this->normalize($expectedFragment), $output);
    $this->assertNoEmptyHeadings($output);
  }

  /**
   * Data provider for testEmptyHeadingsAreReplacedWithParagraphs().
   *
   * @return array
   *   Test cases with input HTML and expected output fragments.
   */
  public static function providerEmptyHeadingsAreReplacedWithParagraphs(): array {
    return [
      'whitespace only' => [
        '<h2>   </h2><p>Next</p>',
        '<p>   </p><p>Next</p>',
      ],
      '&nbsp; only' => [
        '<h3>&nbsp;&nbsp;</h3>',
        '<p>  </p>',
      ],
      'comments only' => [
        '<h4><!-- comment --></h4>',
        '<p><!-- comment --></p>',
      ],
      'empty inline wrappers' => [
        '<h5><em></em><span>   </span></h5>',
        '<p><em></em><span>   </span></p>',
      ],
      'preserve attributes' => [
        '<h2 id="title" class="alpha bravo">  </h2>',
        '<p id="title" class="alpha bravo">  </p>',
      ],
      'preserve children (nested elements retained)' => [
        '<h3 class="h h--x"><span></span><!--c--><i> </i></h3>',
        '<p class="h h--x"><span></span><!--c--><i> </i></p>',
      ],
      'consecutive empty headings' => [
        '<h2> </h2><h3>&nbsp;</h3><div>ok</div>',
        '<p> </p><p> </p><div>ok</div>',
      ],
      'br only' => [
        '<h3><br></h3>',
        '<p><br></p>',
      ],
    ];
  }

  public function testHeadingsWithVisibleTextRemain(): void {
    $input = '<h2>Real Heading</h2><h3><strong>Visible</strong> text</h3>';
    $out = $this->normalize($this->filter->process($input, 'en')->getProcessedText());
    $this->assertStringContainsString('<h2>Real Heading</h2>', $out);
    $this->assertStringContainsString('<h3><strong>Visible</strong> text</h3>', $out);
    $this->assertStringNotContainsString('<p>Real Heading</p>', $out);
  }


  public function testTips(): void {
    // Mock the string translation service.
    /** @var \Drupal\Core\StringTranslation\TranslationInterface|\PHPUnit\Framework\MockObject\MockObject $string_translation */
    $string_translation = $this->createMock('Drupal\Core\StringTranslation\TranslationInterface');
    $string_translation->method('translateString')
      ->willReturn('Converts empty headings (H1–H6) into paragraphs to avoid accessibility issues while keeping layout spacing.');

    $this->filter->setStringTranslation($string_translation);

    $tips = $this->filter->tips();
    $this->assertIsString($tips);
    $this->assertNotSame('', trim($tips));
  }

}
