<?php

declare(strict_types=1);

namespace Drupal\Tests\accessibility_filters\Unit;

use Drupal\accessibility_filters\Plugin\Filter\EmptyLinksFilter;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

#[CoversClass(EmptyLinksFilter::class)]
#[Group('accessibility_filters')]
final class EmptyLinksFilterTest extends UnitTestCase {

  /**
   * The filter plugin instance.
   *
   * @var \Drupal\accessibility_filters\Plugin\Filter\EmptyLinksFilter
   */
  private EmptyLinksFilter $filter;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // The filter has no constructor deps; instantiate directly.
    $this->filter = new EmptyLinksFilter([], 'empty_links_filter', ['provider' => 'accessibility_filters']);
  }

  /**
   * Small normalization to reduce brittle string assertions.
   */
  private function normalize(string $html): string {
    $html = html_entity_decode($html, ENT_QUOTES | ENT_HTML5);
    // NBSP -> space.
    $html = preg_replace('/\x{00A0}/u', ' ', $html) ?? $html;
    // Collapse ws.
    $html = preg_replace('/\s+/', ' ', $html) ?? $html;
    return trim($html);
  }

  /**
   * Assert no empty links remain in the output.
   */
  private function assertNoEmptyLinks(string $html): void {
    $this->assertDoesNotMatchRegularExpression(
      '/<a[^>]*>(\s|&nbsp;|<!--.*?-->|<[^>]*>\s*<\/[^>]*>)*<\/a>/si',
      $html
    );
  }

  #[DataProvider('providerEmptyLinksAreCompletelyRemoved')]
  public function testEmptyLinksAreCompletelyRemoved(string $input, string $expectedFragment): void {
    $result = $this->filter->process($input, 'en');
    $output = $this->normalize($result->getProcessedText());

    $this->assertIsString($output);
    $this->assertStringContainsString($this->normalize($expectedFragment), $output);
    $this->assertNoEmptyLinks($output);
  }

  /**
   * Data provider for testEmptyLinksAreCompletelyRemoved().
   *
   * @return array
   *   Test cases with input HTML and expected output fragments.
   */
  public static function providerEmptyLinksAreCompletelyRemoved(): array {
    return [
      'whitespace only' => [
        '<p><a href="#">   </a>text</p>',
        '<p>text</p>',
      ],
      '&nbsp; only' => [
        '<div><a href="/">&nbsp;&nbsp;</a>content</div>',
        '<div>content</div>',
      ],
      'comments only' => [
        '<span><a><!-- comment --></a>after</span>',
        '<span>after</span>',
      ],
      'empty inline wrappers' => [
        '<p><a href="#"><em></em><span>   </span></a>Text</p>',
        '<p>Text</p>',
      ],
      'attributes do not matter' => [
        '<a href="https://example.com" id="link" class="btn">  </a><div>next</div>',
        '<div>next</div>',
      ],
      'nested empty elements' => [
        '<p><a class="link"><span></span><!--c--><i> </i></a>after</p>',
        '<p>after</p>',
      ],
      'consecutive empty links' => [
        '<a href="#"> </a><a href="#">&nbsp;</a><div>ok</div>',
        '<div>ok</div>',
      ],
      'br only' => [
        '<a href="#"><br></a><span>text</span>',
        '<span>text</span>',
      ],
      'empty link between text' => [
        'Before<a href="#"> </a>After',
        'BeforeAfter',
      ],
    ];
  }

  public function testLinksWithVisibleTextRemain(): void {
    $input = '<a href="#">Real Link</a><a href="/"><strong>Visible</strong> text</a>';
    $out = $this->normalize($this->filter->process($input, 'en')->getProcessedText());
    $this->assertStringContainsString('<a href="#">Real Link</a>', $out);
    $this->assertStringContainsString('<a href="/"><strong>Visible</strong> text</a>', $out);
  }

  public function testTips(): void {
    // Mock the string translation service.
    /** @var \Drupal\Core\StringTranslation\TranslationInterface|\PHPUnit\Framework\MockObject\MockObject $string_translation */
    $string_translation = $this->createMock('Drupal\Core\StringTranslation\TranslationInterface');
    $string_translation->method('translateString')
      ->willReturn('Removes empty anchor (<a>) tags to prevent accessibility issues.');

    $this->filter->setStringTranslation($string_translation);

    $tips = $this->filter->tips();
    $this->assertIsString($tips);
    $this->assertNotSame('', trim($tips));
  }

}
