<?php

namespace Drupal\account_field_split;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class ExtraFieldManager.
 *
 * Split User useless "Username and password" field into extra fields which
 * could be managed separately.
 */
class ExtraFieldManager implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The current route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * Constructs a new ExtraFieldManager.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match service.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user service.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
    RouteMatchInterface $route_match,
    AccountProxyInterface $current_user,
  ) {
    $this->moduleHandler = $module_handler;
    $this->routeMatch = $route_match;
    $this->currentUser = $current_user;
  }

  /**
   * Do form alter.
   */
  public function formAlter(array &$form, FormStateInterface $form_state, $form_id) {
    $build_info = $form_state->getBuildInfo();
    if ((empty($build_info['base_form_id']) || 'user_form' != $build_info['base_form_id']) && 'user_form' != $form_id) {
      return;
    }

    // Skip field splitting on password reset forms for compatibility with
    // simple_pass_reset module.
    if ($this->moduleHandler->moduleExists('simple_pass_reset')) {
      $route_name = $this->routeMatch->getRouteName();

      if ($route_name === 'user.reset' && $this->currentUser->isAnonymous()) {
        return;
      }
    }

    if (isset($form['account']) && is_array($form['account'])) {
      $account_elements = Element::children($form['account']);
      foreach ($account_elements as $account_element) {
        if (isset($form['account'][$account_element])) {
          $form[$account_element] = $form['account'][$account_element];
          unset($form['account'][$account_element]);
        }
      }
    }
  }

  /**
   * Return list of extra fields.
   *
   * @return array
   *   Array of extra fields description.
   */
  public function extraFieldInfo() {
    $extra = [];
    $fields = [
      'mail' => $this->t('E-mail address'),
      'name' => $this->t('Username'),
      'pass' => $this->t('Password'),
      'status' => $this->t('Status'),
      'roles' => $this->t('Roles'),
      'notify' => $this->t('Notify user about new account'),
      'current_pass' => $this->t('Current password'),
    ];
    $description = $this->t('User profile element');

    foreach ($fields as $field => $label) {
      $extra['user']['user']['form'][$field] = [
        'label' => $label,
        'description' => $description,
        'weight' => 0,
        'visible' => TRUE,
      ];
    }

    return $extra;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('module_handler'),
      $container->get('current_route_match'),
      $container->get('current_user')
    );
  }

}
