<?php

namespace Drupal\acquia_cms_common\Commands;

use Consolidation\AnnotatedCommand\CommandData;
use Consolidation\AnnotatedCommand\CommandError;
use Drupal\acquia_cms_common\Services\AcmsUtilityService;
use Drupal\acquia_cms_common\Services\ConfigImporterService;
use Drupal\acquia_cms_site_studio\Facade\CohesionFacade;
use Drupal\config\StorageReplaceDataWrapper;
use Drupal\Core\Config\FileStorage;
use Drupal\Core\Config\StorageComparer;
use Drupal\Core\Config\StorageInterface;
use Drupal\Core\DependencyInjection\ClassResolver;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drush\Commands\DrushCommands;
use Drush\Exceptions\UserAbortException;
use Symfony\Component\Console\Question\ChoiceQuestion;

/**
 * A Drush command file.
 *
 * This files contains custom drush command to provide a way to import
 * standard configuration with partial option along with site studio package
 * for particular some or all given modules.
 */
final class AcmsConfigImportCommands extends DrushCommands {

  /**
   * The allowed scope.
   *
   * @var string[]
   * Allowed scope for drush commands.
   */
  protected $allowedScope = ['config'];

  /**
   * The config manager.
   *
   * @var \Drupal\Core\Config\ConfigManagerInterface
   */
  protected $configManager;

  /**
   * The StorageInterface object.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $configStorage;

  /**
   * The acquia_cms_common.config.importer service object.
   *
   * @var \Drupal\acquia_cms_common\Services\ConfigImporterService
   */
  protected $configImporter;

  /**
   * Holds the class name for ConfigCommands.
   *
   * @var string
   */
  protected $configCommands;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The string translation interface.
   *
   * @var \Drupal\Core\StringTranslation\TranslationInterface
   */
  protected $stringTranslation;

  /**
   * Cohesion Facade class object.
   *
   * @var \Drupal\acquia_cms_site_studio\Facade\CohesionFacade
   */
  protected $cohesionFacade;

  /**
   * The acquia cms utility service.
   *
   * @var \Drupal\acquia_cms_common\Services\AcmsUtilityService
   */
  protected $acmsUtilityService;

  /**
   * Get config storage object.
   *
   * @return \Drupal\Core\Config\StorageInterface
   *   The StorageInterface.
   */
  public function getConfigStorage() {
    return $this->configStorage;
  }

  /**
   * Get string translation object.
   *
   * @return \Drupal\Core\StringTranslation\TranslationInterface
   *   The TranslationInterface.
   */
  public function getStringTranslation() {
    return $this->stringTranslation;
  }

  /**
   * Get module handler object.
   *
   * @return \Drupal\Core\Extension\ModuleHandlerInterface
   *   The ModuleHandlerInterface.
   */
  public function getModuleHandler() {
    return $this->moduleHandler;
  }

  /**
   * The class constructor.
   *
   * @param \Drupal\acquia_cms_common\Services\ConfigImporterService $configImporterService
   *   The acquia_cms_common.config.importer service.
   * @param \Drupal\Core\Config\StorageInterface $configStorage
   *   The StorageInterface.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   The TranslationInterface.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The ModuleHandlerInterface.
   * @param \Drupal\Core\DependencyInjection\ClassResolver $classResolver
   *   The class resolver.
   * @param \Drupal\acquia_cms_common\Services\AcmsUtilityService $acmsUtilityService
   *   The acquia cms service.
   */
  public function __construct(
    ConfigImporterService $configImporterService,
    StorageInterface $configStorage,
    TranslationInterface $stringTranslation,
    ModuleHandlerInterface $moduleHandler,
    ClassResolver $classResolver,
    AcmsUtilityService $acmsUtilityService,
  ) {
    parent::__construct();
    $this->configStorage = $configStorage;
    $this->stringTranslation = $stringTranslation;
    $this->moduleHandler = $moduleHandler;
    if ($this->moduleHandler->moduleExists('acquia_cms_site_studio')) {
      $this->cohesionFacade = $classResolver->getInstanceFromDefinition(CohesionFacade::class);
      $this->allowedScope = ['config', 'site-studio', 'all'];
    }
    $this->acmsUtilityService = $acmsUtilityService;
    $this->configImporter = $configImporterService;
    if (class_exists("Drush\Drupal\Commands\config\ConfigCommands")) {
      $this->configCommands = "Drush\Drupal\Commands\config\ConfigCommands";
    }
    else {
      $this->configCommands = "Drush\Commands\config\ConfigCommands";
    }
  }

  /**
   * Reset configurations to default.
   *
   * Command to reset configuration for ACMS modules
   * to the default canonical config, as exported in code.
   *
   * @param array $package
   *   The name of modules separated by space.
   * @param array $options
   *   The options array.
   *
   * @option scope
   *   The scope for particular package to be imported.
   * @option delete-list
   *   The comma separated list of config files to be deleted during import.
   * @command acms:config-reset
   * @aliases acr
   * @usage acms:config-reset
   *   Reset the configuration to the default.
   * @usage acms:config-reset acquia_cms_article acquia_cms_person --scope=all
   * --delete-list=search_api.index.acquia_search_index
   *   Reset the configuration to the default.
   *
   * @throws \Drush\Exceptions\UserAbortException
   */
  public function resetConfigurations(
    array $package,
    array $options = [
      'scope' => NULL,
      'delete-list' => NULL,
    ],
  ) {
    $this->io()->text(["Welcome to the Acquia CMS config reset wizard.",
      "This should be used with extreme caution and can lead to unexpected behavior on your site if not well tested.",
      "Do not run this in production until you've tested it in a safe, non-public environment first.",
    ]);
    // Reset the configurations for given packages aka modules
    // package, scope & delete-list are being added in validate command.
    $this->doImport($package, $options['scope'], $options['delete-list']);
  }

  /**
   * Get package from user input if not provided already.
   *
   * @return array
   *   The package from user input.
   *
   * @throws \Drush\Exceptions\UserAbortException
   */
  private function getPackagesFromUserInput(): array {
    // Let's get input from user if not provided package with command.
    $acms_modules = $this->filterModuleForConfig();
    $question_string = 'Choose a module that needs a configuration reset. Separate multiple choices with commas, e.g. "1,2,4".';
    $question = $this->createMultipleChoiceOptions($question_string, $acms_modules);
    $types = $this->io()->askQuestion($question);
    if (in_array('Cancel', $types)) {
      throw new UserAbortException();
    }
    elseif (in_array('All', $types)) {
      $package = $acms_modules;
    }
    else {
      $package = $types;
    }
    return $package;
  }

  /**
   * Filter out those modules which do not have config to import from the list.
   *
   * @return array
   *   The list of module which has configurations.
   */
  private function filterModuleForConfig(): array {
    $acms_modules = $this->getAcmsModules();
    $acms_filtered_modules = [];
    foreach ($acms_modules as $module) {
      $dir = $this->moduleHandler->getModule($module)->getPath();
      $install = "$dir/config/install";
      $optional = "$dir/config/optional";
      if (is_dir($install) || is_dir($optional)) {
        $acms_filtered_modules[] = $module;
      }
    }

    return $acms_filtered_modules;
  }

  /**
   * Get list of Acquia CMS modules.
   *
   * @return array
   *   Array of acms modules.
   */
  private function getAcmsModules(): array {
    $acms_modules = [];
    $acms_extensions = $this->acmsUtilityService->getAcquiaCmsModuleList();
    foreach ($acms_extensions as $key => $module) {
      if ($module->getType() === 'module') {
        $acms_modules[] = $key;
      }
    }
    return $acms_modules;
  }

  /**
   * Create multiple choice question.
   *
   * @param string $question_string
   *   The question to ask.
   * @param array $choice_options
   *   The choice of options.
   * @param int|null $default
   *   The default option in multi-choice.
   *
   * @return \Symfony\Component\Console\Question\ChoiceQuestion
   *   The ChoiceQuestion
   */
  private function createMultipleChoiceOptions(string $question_string, array $choice_options, $default = NULL): ChoiceQuestion {
    $choices = array_merge(['Cancel'], $choice_options);
    array_push($choices, 'All');
    $question = new ChoiceQuestion(dt($question_string), $choices, $default);
    $question->setMultiselect(TRUE);
    return $question;
  }

  /**
   * Import configuration based on scope.
   *
   * @param array $package
   *   The array of modules.
   * @param string $scope
   *   The scope.
   * @param array $delete_list
   *   The list of config files to be deleted during import.
   *
   * @throws \Drush\Exceptions\UserAbortException
   * @throws \Exception
   */
  private function doImport(array $package, string $scope, array $delete_list) {
    $config_files = $modules = [];
    if (in_array($scope, ['config', 'all'])) {
      foreach ($package as $module) {
        $config_files = array_merge($config_files, $this->getConfigFiles($module));
      }
      // Validate delete list against given scope of configurations.
      if (!$this->validDeleteList($config_files, $delete_list)) {
        throw new \Exception("The file specified in --delete-list option is invalid.");
      }
      $this->importPartialConfig($config_files, $delete_list);
    }

    // Build site studio packages.
    if (in_array($scope, ['site-studio', 'all'])) {
      foreach ($package as $module_name) {
        $modules[$module_name] = $this->moduleHandler->getModule($module_name);
      }
      $ss_packages = $this->cohesionFacade->buildPackageList($modules);
      // Confirm the site studio changes before import.
      if ($this->buildSiteStudioChangeList($ss_packages)) {
        if (!$this->io()->confirm(dt('Are you sure you want to reset these site studio packages?'))) {
          throw new UserAbortException();
        }
        // Import the site studio configurations.
        if ($this->cohesionFacade->importSiteStudioPackages($ss_packages)) {
          drush_backend_batch_process();
        }
      }
      else {
        $this->io()->success('No site studio package to import.');
      }
    }
  }

  /**
   * Show change list for site studio packages.
   *
   * @param array $ss_config_files
   *   Array of configurations file.
   *
   * @return bool
   *   The package status.
   */
  private function buildSiteStudioChangeList(array $ss_config_files): bool {
    if (empty($ss_config_files)) {
      return FALSE;
    }
    $rows = [];
    foreach ($ss_config_files as $package) {
      $rows[] = [$package['source']['module_name'], $package['source']['path']];
    }
    // Show warning if site-studio is in scope.
    $this->io()->warning("This can have unintended side effects for existing pages built using previous versions of components, it might literally break them, and should be tested in a non-production environment first.");
    $this->io()->table(['Module name', 'Package path'], $rows);
    return TRUE;
  }

  /**
   * Import configurations for the given module & its scope.
   *
   * @param string $module
   *   The name of module.
   *
   * @return array
   *   The array of config files.
   */
  private function getConfigFiles(string $module): array {
    $config_files = [];
    $module_path = $this->moduleHandler->getModule($module)->getPath();
    $source_install = $module_path . '/config/install';
    $source_optional = $module_path . '/config/optional';

    // Get optional configuration list for specified module.
    if (file_exists($source_optional)) {
      $source_storage_dir = $this->configCommands::getDirectory($source_optional);
      $source_storage = new FileStorage($source_storage_dir);
      foreach ($source_storage->listAll() as $name) {
        $config_files[$name] = $source_storage->read($name);
      }
      // Remove configuration files where its dependencies cannot be met
      // in case of optional configurations.
      $this->removeDependentFiles($config_files);
    }

    // Now get default configurations.
    if (file_exists($source_install)) {
      $source_storage_dir = $this->configCommands::getDirectory($source_install);
      $source_storage = new FileStorage($source_storage_dir);
      foreach ($source_storage->listAll() as $name) {
        $config_files[$name] = $source_storage->read($name);
      }
    }

    return $config_files;
  }

  /**
   * Remove configuration files where its dependencies cannot be met.
   *
   * @param array $config_files
   *   Array of configurations files.
   */
  private function removeDependentFiles(array &$config_files) {
    $enabled_extensions = $this->acmsUtilityService->getEnabledExtensions();
    $all_config = $this->getConfigStorage()->listAll();
    $all_config = array_combine($all_config, $all_config);
    foreach ($config_files as $config_name => $data) {
      // Remove configuration where its dependencies cannot be met.
      $remove = !$this->acmsUtilityService->validateDependencies($config_name, $data, $enabled_extensions, $all_config);
      if ($remove) {
        unset($config_files[$config_name]);
      }
    }
  }

  /**
   * Import configurations for the given sources.
   *
   * @param array $config_files
   *   The config file that needs re-import.
   * @param array $delete_list
   *   The list of configurations to be deleted before import.
   *
   * @throws \Drush\Exceptions\UserAbortException
   * @throws \Exception
   */
  private function importPartialConfig(array $config_files, array $delete_list) {
    // Determine $source_storage in partial case.
    $active_storage = $this->getConfigStorage();

    $replacement_storage = new StorageReplaceDataWrapper($active_storage);
    foreach ($config_files as $name => $data) {
      // We should not re-import cohesion settings,
      // it will override the site studio credentials which
      // will break the whole site. Also re-importing
      // search_api.index.content will have unexpected error since
      // we have modified it using facade to add index field.
      if ($name === 'cohesion.settings' || $name === 'search_api.index.content') {
        continue;
      }
      $replacement_storage->replaceData($name, $data);
    }
    $source_storage = $replacement_storage;
    // In case of --delete-list option lets delete configurations from
    // source storage before running the actual importing.
    if ($delete_list) {
      foreach ($delete_list as $del_config_item) {
        // Allow for accidental .yml extension.
        if (substr($del_config_item, '.yml')) {
          $del_config_item = substr($del_config_item, 0, -4);
        }
        if ($source_storage->exists($del_config_item)) {
          $source_storage->delete($del_config_item);
        }
      }
    }
    $storage_comparer = new StorageComparer($source_storage, $active_storage);
    if (!$storage_comparer->createChangelist()->hasChanges()) {
      $this->logger()->notice(('There are no changes to import.'));
      exit();
    }

    // List the changes in table format.
    $change_list = [];
    foreach ($storage_comparer->getAllCollectionNames() as $collection) {
      $change_list[$collection] = $storage_comparer->getChangelist(NULL, $collection);
    }
    $table = $this->configCommands::configChangesTable($change_list, $this->output());
    $table->render();
    $this->io()->warning("Any overridden configurations will be reverted back with the one listed above which may result in unexpected behaviour.");
    if (!$this->io()->confirm(dt('Import these configuration changes?'))) {
      throw new UserAbortException();
    }
    $this->configImporter->doImport($storage_comparer, $this->logger());
  }

  /**
   * Hook validate for acms config reset command.
   *
   * @hook validate acms:config-reset
   *
   * @throws \Drush\Exceptions\UserAbortException
   */
  public function validateConfigResetCommand(CommandData $commandData) {
    // Since we are running config import with partial option
    // Lets check config module is enabled or not.
    if (!$this->moduleHandler->moduleExists('config')) {
      return new CommandError('Config module is not enabled, please enable it.');
    }

    $messages = [];
    $isInteractive = $commandData->input()->isInteractive();
    $scope = $commandData->input()->getOption('scope');
    $delete_list = $commandData->input()->getOption('delete-list');
    $package = $commandData->input()->getArgument('package');

    if (isset($scope) && !in_array($scope, $this->allowedScope)) {
      $messages[] = 'Invalid scope, allowed values are [config, site-studio, all]';
    }
    if ($package && !$this->hasValidPackage($package)) {
      $messages[] = 'Given packages are not valid, try providing a list of ACMS modules separated by space ex: acquia_cms_article acquia_cms_place';
    }
    // In case of --delete-list option.
    if ($delete_list) {
      $delete_list_array = array_filter(explode(',', $delete_list));
      if (empty($delete_list_array)) {
        $messages[] = dt("The file specified in --delete-list option is in the wrong format.");
      }
      else {
        $commandData->input()->setOption('delete-list', $delete_list_array);
      }
    }
    else {
      $commandData->input()->setOption('delete-list', []);
    }
    // In case of -y lets check user has provided all the required arguments.
    if (!$isInteractive && (!$package || !$scope)) {
      $messages[] = 'In order to use -y option, please provide a package and scope variable.';
    }
    // Get packages from user input.
    if ($isInteractive && empty($messages) && !$package) {
      $package = $this->getPackagesFromUserInput();
      $commandData->input()->setArgument('package', $package);
    }
    // Get scope from user input.
    if ($isInteractive && empty($messages) && !$scope) {
      $scope = $this->io()->choice(dt('Choose a scope.'), $this->allowedScope, NULL);
      $commandData->input()->setOption('scope', $this->allowedScope[$scope]);
    }
    if ($messages) {
      return new CommandError(implode(' ', $messages));
    }
  }

  /**
   * Validate the given delete list has valid configuration file.
   *
   * @param array $config_file
   *   The list of configurations file per scope.
   * @param array $delete_list_array
   *   The list of config file to be deleted.
   *
   * @return bool
   *   Boolean indicate true, false.
   */
  private function validDeleteList(array $config_file, array $delete_list_array): bool {
    $config_file_list = array_keys($config_file);
    $valid = TRUE;
    foreach ($delete_list_array as $config_name) {
      if (!in_array($config_name, $config_file_list)) {
        $valid = FALSE;
        break;
      }
    }
    return $valid;
  }

  /**
   * Check the provided packages are valid.
   *
   * @param array $packages
   *   The array of package.
   *
   * @return bool
   *   The status of package.
   */
  private function hasValidPackage(array $packages): bool {
    $valid_package = $this->getAcmsModules();
    foreach ($packages as $package) {
      if (!in_array($package, $valid_package)) {
        return FALSE;
      }
    }
    return TRUE;
  }

}
