<?php

namespace Drupal\acquia_cms_support\Service;

use Drupal\Component\Diff\Diff;
use Drupal\Component\Serialization\Yaml;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\FileStorage;
use Drupal\Core\Config\ImportStorageTransformer;
use Drupal\Core\Config\InstallStorage;
use Drupal\Core\Config\StorageComparer;
use Drupal\Core\Config\StorageInterface;

/**
 * Defines a service which provides config sync for acquia cms.
 *
 * @internal
 *   This is a totally internal part of Acquia CMS and may be changed in any
 *   way, or removed outright, at any time without warning. External code should
 *   not use this class!
 */
class AcquiaCmsConfigSyncService {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The target storage.
   *
   * @var \Drupal\Core\Config\StorageInterface
   */
  protected $targetStorage;

  /**
   * The import transformer service.
   *
   * @var \Drupal\Core\Config\ImportStorageTransformer
   */
  protected $importTransformer;

  /**
   * AcquiaCmsConfigSyncService constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\StorageInterface $target_storage
   *   The target storage.
   * @param \Drupal\Core\Config\ImportStorageTransformer $import_transformer
   *   The import transformer service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    StorageInterface $target_storage,
    ImportStorageTransformer $import_transformer,
  ) {
    $this->configFactory = $config_factory;
    $this->targetStorage = $target_storage;
    $this->importTransformer = $import_transformer;
  }

  /**
   * Get the parity between Acquia CMS vs Active configuration.
   *
   * @param string $configFile
   *   Configuration file.
   * @param \Drupal\Core\Config\StorageInterface $syncStorage
   *   The storage to use as sync storage for compairing changes.
   *
   * @return int
   *   Parity value between configuration.
   */
  public function getParity($configFile, StorageInterface $syncStorage): int {
    // Database configuration.
    $activeConfiguration = explode("\n", Yaml::encode($this->targetStorage->read($configFile)));
    // Configuration in files.
    $originalConfiguration = explode("\n", Yaml::encode($syncStorage->read($configFile)));
    $activeConfiguration = $this->removeNonRequiredKeys($activeConfiguration);
    $diff = new Diff($originalConfiguration, $activeConfiguration);
    $totalLines = count($activeConfiguration);
    $editedLines = 0;
    $editDiffConfig = $diff->getEdits();
    if (!empty($editDiffConfig)) {
      foreach ($editDiffConfig as $diffOp) {
        if ($diffOp->type !== 'copy') {
          $editedLines += is_array($diffOp->closing) ? count($diffOp->closing) : 0;
        }
      }
    }

    return 100 - (int) round($editedLines / $totalLines * 100, 0);
  }

  /**
   * Remove _core, uuid, default_config_hash from configurations.
   *
   * @param array $data
   *   Configuration data.
   *
   * @return array
   *   Array of configurations after removing keys.
   */
  public function removeNonRequiredKeys(array $data) {
    // Remove the _core, uuid, default_config_hash from the configuration.
    return array_values(array_filter(
      $data,
      function ($val) {
        return (strpos($val, '_core') !== 0) && (strpos(trim($val), 'default_config_hash:') !== 0) && (strpos($val, 'uuid:') !== 0);
      },
      ARRAY_FILTER_USE_BOTH
    ));
  }

  /**
   * Get install config directory storage.
   *
   * @param string $path
   *   Path to use for install filestorage.
   *
   * @return object
   *   File Storage Object.
   */
  public function getInstallStorage($path) {
    return $this->getFileStorage($path . '/' . InstallStorage::CONFIG_INSTALL_DIRECTORY);
  }

  /**
   * Get optional config directory storage.
   *
   * @param string $path
   *   Path to use for optional filestorage.
   *
   * @return object
   *   File Storage Object.
   */
  public function getOptionalStorage($path) {
    return $this->getFileStorage($path . '/' . InstallStorage::CONFIG_OPTIONAL_DIRECTORY);
  }

  /**
   * Get the storage for a given path.
   *
   * @param string $path
   *   Path to use for filestorage.
   *
   * @return object
   *   FileStorage Object.
   */
  private function getFileStorage($path) {
    return new FileStorage($path);
  }

  /**
   * List all the changed (create and update) config from a storage.
   *
   * @param \Drupal\Core\Config\StorageInterface $syncStorage
   *   The storage to use as sync storage for compairing changes.
   *
   * @return array
   *   List of the chaged config.
   */
  public function getOverriddenConfig(StorageInterface $syncStorage) {
    $overriddenConfig = [];
    $storageComparer = $this->getStoragecomparer($syncStorage)->createChangelist();
    if ($storageComparer->hasChanges()) {
      $changedConfig = $this->getCreateAndUpdateChangeList($storageComparer);
      foreach ($changedConfig as $config) {
        $parity = $this->getParity($config, $syncStorage);
        if ($parity === 100) {
          continue;
        }
        $overriddenConfig[] = [
          'name' => $config,
          'parity' => $parity,
        ];
      }
    }
    return $overriddenConfig;
  }

  /**
   * Get Unchanged config list.
   *
   * @param \Drupal\Core\Config\StorageInterface $syncStorage
   *   The storage to use as sync storage for compairing changes.
   *
   * @return \Drupal\Core\Config\StorageComparer
   *   Storage Comparer object.
   */
  private function getStoragecomparer(StorageInterface $syncStorage) {
    return new StorageComparer($syncStorage, $this->targetStorage);
  }

  /**
   * Get changed config list.
   *
   * @param \Drupal\Core\Config\StorageComparer $storageComparer
   *   The storage to use as sync storage for compairing changes.
   *
   * @return array
   *   Array of changed configurations.
   */
  private function getCreateAndUpdateChangeList(StorageComparer $storageComparer) {
    $changedConfig = [];
    $createdConfig = $storageComparer->getChangelist('create');
    $updatedConfig = $storageComparer->getChangelist('update');
    $changedConfig = \array_merge($changedConfig, $createdConfig, $updatedConfig);
    return $changedConfig;
  }

  /**
   * Get Unchanged config list.
   *
   * @param \Drupal\Core\Config\StorageInterface $syncStorage
   *   The storage to use as sync storage for compairing changes.
   *
   * @return array
   *   Array of unchanged configurations.
   */
  public function getUnChangedConfig(StorageInterface $syncStorage) {
    $unChangedConfigList = [];
    $storageComparer = $this->getStoragecomparer($syncStorage)->createChangelist();
    if ($storageComparer->hasChanges()) {
      $changeList = $this->getCreateAndUpdateChangeList($storageComparer);
      foreach ($changeList as $config) {
        $parity = $this->getParity($config, $syncStorage);
        if ($parity !== 100) {
          continue;
        }
        \array_push($unChangedConfigList, $config);
      }
    }
    else {
      $unChangedConfigList = $syncStorage->listAll();
    }
    return $unChangedConfigList;
  }

}
