<?php

namespace Drupal\Tests\acquia_cms_event\ExistingSite;

use Behat\Mink\Element\ElementInterface;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\Tests\block\Traits\BlockCreationTrait;
use weitzman\DrupalTestTraits\ExistingSiteBase;

/**
 * Tests breadcrumbs generated from content type sub-types.
 *
 * @group acquia_cms_event
 * @group acquia_cms
 * @group low_risk
 * @group pr
 * @group push
 */
class BreadcrumbTest extends ExistingSiteBase {

  use BlockCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $block = $this->placeBlock('system_breadcrumb_block', [
      'region' => 'content',
    ]);
    $this->markEntityForCleanup($block);
  }

  /**
   * Data provider for ::testBreadcrumb().
   *
   * @return array[]
   *   Sets of arguments to pass to the test method.
   */
  public static function providerBreadcrumb() : array {
    return [
      [
        'event',
        'Party',
        [
          ['Events', '/events'],
          ['Party', '/events/type/party'],
        ],
      ],
    ];
  }

  /**
   * Tests the breadcrumbs generated for sub-types of a content type.
   *
   * @param string $node_type
   *   The content type under test.
   * @param string $sub_type
   *   The label of the sub-type taxonomy term to generate.
   * @param array[] $expected_breadcrumb
   *   The expected breadcrumb links, in their expected order. Each element
   *   should be a tuple containing the text of the link, and its target path.
   *
   * @dataProvider providerBreadcrumb
   */
  public function testBreadcrumb(string $node_type, string $sub_type, array $expected_breadcrumb) : void {
    $vocabulary = Vocabulary::load($node_type . '_type');
    $sub_type = $this->createTerm($vocabulary, ['name' => $sub_type]);

    $node = $this->createNode([
      'type' => $node_type,
      'field_' . $vocabulary->id() => $sub_type->id(),
      'moderation_state' => 'published',
    ]);
    // The breadcrumb should always have the unlinked node title at the end.
    array_push($expected_breadcrumb, $node->getTitle());

    $this->drupalGet($node->toUrl());
    $this->assertSession()->statusCodeEquals(200);
    if ($this->container->get('module_handler')->moduleExists('acquia_cms_search')) {
      $this->assertBreadcrumb($expected_breadcrumb);
    }
  }

  /**
   * Data provider for ::testBreadcrumbNoSubType().
   *
   * @return array[]
   *   Sets of arguments to pass to the test method.
   */
  public static function providerNoSubType() : array {
    $object = new self('test');
    $map = function (array $arguments) : array {
      unset($arguments[1], $arguments[2][1]);
      return $arguments;
    };
    return array_map($map, $object->providerBreadcrumb());
  }

  /**
   * Tests the breadcrumbs generated content without a sub-type.
   *
   * @param string $node_type
   *   The content type under test.
   * @param array[] $expected_breadcrumb
   *   The expected breadcrumb links, in their expected order. Each element
   *   should be a tuple containing the text of the link, and its target path.
   *
   * @dataProvider providerNoSubType
   */
  public function testBreadcrumbNoSubType(string $node_type, array $expected_breadcrumb) : void {
    $node = $this->createNode([
      'type' => $node_type,
      'moderation_state' => 'published',
    ]);
    // Resetting the keys so that proper key order is maintained.
    $expected_breadcrumb = array_values($expected_breadcrumb);
    // The breadcrumb should always have the unlinked node title at the end.
    array_push($expected_breadcrumb, $node->getTitle());

    $this->drupalGet($node->toUrl());
    $this->assertSession()->statusCodeEquals(200);
    if ($this->container->get('module_handler')->moduleExists('acquia_cms_search')) {
      $this->assertBreadcrumb($expected_breadcrumb);
    }
  }

  /**
   * Asserts the presence of a set of breadcrumb links.
   *
   * @param array $expected_breadcrumb
   *   The expected breadcrumb items, in their expected order. Each element
   *   should either be a tuple containing the text of the link and its target
   *   path, or a string with the text of the item (if the item is not a link).
   */
  private function assertBreadcrumb(array $expected_breadcrumb) : void {
    $assert_session = $this->assertSession();

    // Create an array of tuples containing the text and target path of every
    // breadcrumb link.
    $map = function (ElementInterface $list_item) {
      $link = $list_item->find('css', 'a');
      if ($link) {
        return [
          $link->getText(),
          // Let remove -tid from the link generated by facet pretty path.
          preg_replace('/-[^-]*$/', '', $link->getAttribute('href')),
        ];
      }
      else {
        return $list_item->getText();
      }
    };
    $breadcrumb = array_map($map, $assert_session->elementExists('css', 'h2#system-breadcrumb + ol')->findAll('css', 'li'));
    $assert_session->statusCodeEquals(200);
    $this->assertSame($expected_breadcrumb, $breadcrumb);
  }

}
