<?php

declare(strict_types=1);

namespace Drupal\acquia_cms_image\Hook;

use Drupal\acquia_cms_image\SiteLogo;
use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides entity module hooks for the Acquia CMS Image module.
 *
 * @internal
 */
class ModuleHooks implements ContainerInjectionInterface {

  /**
   * Constructs a new ModuleHooks instance.
   *
   * @param \Drupal\Core\DependencyInjection\ClassResolverInterface $classResolver
   *   The class resolver service.
   */
  public function __construct(
    private readonly ClassResolverInterface $classResolver,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('class_resolver'),
    );
  }

  /**
   * Implements hook_modules_installed().
   *
   * This hook is triggered when modules are installed.
   *
   * @param array $modules
   *   An array of module names that were installed.
   * @param bool $is_syncing
   *   A boolean indicating if the installation is part of configuration sync.
   */
  #[Hook('modules_installed')]
  public function modulesInstalled(array $modules, bool $is_syncing): void {
    // Skip processing during configuration synchronization to avoid conflicts.
    if ($is_syncing) {
      return;
    }

    if (in_array('acquia_cms_image', $modules)) {
      $this->classResolver->getInstanceFromDefinition(SiteLogo::class)
        ->createLogo()->setLogo();
    }
  }

}
