<?php

namespace Drupal\Tests\acquia_cms_image\FunctionalJavascript;

use Behat\Mink\Element\ElementInterface;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\image\Kernel\ImageFieldCreationTrait;

/**
 * Tests embedding Image media in CKEditor.
 *
 * @group acquia_cms_image
 */
class ImageEmbedTest extends WebDriverTestBase {

  use ImageFieldCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['acquia_cms_image'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);
  }

  /**
   * Tests that image field with focal point functionality works correctly.
   *
   * This test verifies:
   * - Image field creation with focal point widget
   * - File upload functionality
   * - Focal point indicator visibility after upload.
   */
  public function testEmbedMedia() {
    $field_name = 'image';

    // Create an image field on the article content type with single
    // cardinality and optional alt text to simplify testing.
    $this->createImageField(
      $field_name,
      'node',
      'article',
      ['cardinality' => 1],
      ['alt_field_required' => 0],
    );

    // Create and login a user.
    $this->drupalLogin($this->drupalCreateUser([
      'access content',
      'create article content',
    ]));

    $this->drupalGet('node/add/article');

    // Configure the form display to use focal point widget instead of default
    // This needs to be done after navigation to ensure proper form context.
    $this->container->get('entity_display.repository')
      ->getFormDisplay('node', 'article', 'default')
      ->setComponent($field_name, [
        'type' => 'image_focal_point',
        'settings' => [
          'preview_image_style' => 'thumbnail',
          'preview_link' => TRUE,
    // Default focal point center.
          'offsets' => '50,50',
          'progress_indicator' => 'throbber',
        ],
      ])
      ->save();

    // Use a reliable test image from Drupal core fixtures.
    $image_file_path = \Drupal::root() . '/core/tests/fixtures/files/image-1.png';

    // Upload the image file to the field.
    $this->getSession()
      ->getPage()
      ->attachFileToField('image', $image_file_path);

    // Wait for the focal point widget to appear after file upload.
    $session = $this->assertSession();
    $element = $session->waitForElementVisible('css', '.field--name-image .focal-point-wrapper');
    $this->assertInstanceOf(ElementInterface::class, $element);

    // Verify the focal point indicator is visible and interactive.
    $indicator = $element->find('css', '.focal-point-indicator');
    $this->assertNotNull($indicator, 'Focal point indicator should be present');
    $this->assertTrue($indicator->isVisible(), 'Focal point indicator should be visible');
  }

}
