<?php

declare(strict_types=1);

namespace Drupal\acquia_cms_image\Hook;

use Drupal\acquia_cms_image\ImageModuleInstaller;
use Drupal\Core\DependencyInjection\ClassResolverInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\editor\EditorInterface;
use Drupal\filter\FilterFormatInterface;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides entity presave hooks for the Acquia CMS Image module.
 *
 * @internal
 */
class EntityPresaveHooks implements ContainerInjectionInterface {

  /**
   * Constructs a new EntityPresaveHooks instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\DependencyInjection\ClassResolverInterface $classResolver
   *   The class resolver service.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly ClassResolverInterface $classResolver,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('class_resolver'),
    );
  }

  /**
   * Responds to entity presave events.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being saved.
   */
  #[Hook('entity_presave')]
  public function entityPresave(EntityInterface $entity): void {
    // Skip processing during configuration synchronization to avoid conflicts.
    if ($entity->isSyncing()) {
      return;
    }

    // Handle filter format and editor configuration.
    if (in_array($entity->id(), ['filtered_html', 'full_html'])) {
      $this->configureTextFormats($entity);
    }

    // Handle user role permission grants.
    if ($entity instanceof RoleInterface && in_array($entity->id(), [
      'content_author',
      'content_editor',
    ])) {
      $this->getImageModuleInstaller()->grantPermissionsToRole($entity);
    }
  }

  /**
   * Configures text formats and their associated editors.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being processed (filter format or editor).
   */
  private function configureTextFormats(EntityInterface $entity): void {
    if ($entity instanceof FilterFormatInterface) {
      $this->configureFilterFormat($entity);
    }
    elseif ($entity instanceof EditorInterface) {
      $this->configureEditor($entity);
    }
  }

  /**
   * Configures a filter format for image media support.
   *
   * @param \Drupal\filter\FilterFormatInterface $entity
   *   The filter format entity.
   */
  private function configureFilterFormat(FilterFormatInterface $entity): void {
    if (!$this->isNewEntity($entity, 'filter_format')) {
      return;
    }

    $this->getImageModuleInstaller()->alterFilterFormat($entity);
  }

  /**
   * Configures an editor for image media support.
   *
   * @param \Drupal\editor\EditorInterface $entity
   *   The editor entity.
   */
  private function configureEditor(EditorInterface $entity): void {
    if (!$this->isNewEntity($entity, 'editor')) {
      return;
    }

    $this->getImageModuleInstaller()->alterEditor($entity);
  }

  /**
   * Checks if an entity is truly new by comparing with database state.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to check.
   * @param string $entityType
   *   The entity type ID.
   */
  private function isNewEntity(EntityInterface $entity, string $entityType): bool {
    $storage = $this->entityTypeManager->getStorage($entityType);
    $existing = $storage->loadUnchanged($entity->id());
    return !$existing;
  }

  /**
   * Gets the ImageModuleInstaller service instance.
   */
  private function getImageModuleInstaller(): ImageModuleInstaller {
    return $this->classResolver->getInstanceFromDefinition(ImageModuleInstaller::class);
  }

}
