<?php

declare(strict_types=1);

namespace Drupal\acquia_cms_image;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\editor\EditorInterface;
use Drupal\filter\FilterFormatInterface;
use Drupal\user\RoleInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles image module installation operations.
 *
 * This service is responsible for:
 * - Granting image-related permissions to content roles
 * - Configuring filter formats to enable media embedding
 * - Setting up CKEditor instances to support image media.
 *
 * @internal
 */
class ImageModuleInstaller implements ContainerInjectionInterface {

  /**
   * Constructs a new ImageModuleInstaller.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service for acquia_cms_image.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
    protected LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('logger.factory')->get('acquia_cms_image')
    );
  }

  /**
   * Executes all installation operations.
   *
   * This method orchestrates the complete setup process:
   * 1. Grants image permissions to appropriate user roles
   * 2. Configures filter formats and editors for image media support.
   *
   * @throws \Exception
   *   If critical installation steps fail.
   */
  public function install(): void {
    $this->grantImagePermissions();
    $this->configureFilters();
  }

  /**
   * Grants image permissions to content roles.
   *
   * Iterates through predefined role-permission mappings and applies
   * them to existing roles in the system.
   */
  private function grantImagePermissions(): void {
    $rolePermissions = $this->getDefaultRolePermissions();

    foreach ($rolePermissions as $roleId => $permissions) {
      try {
        $role = $this->entityTypeManager->getStorage('user_role')->load($roleId);
        if ($role instanceof RoleInterface) {
          $this->grantPermissionsToRole($role, $permissions);
          $role->save();
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Failed to grant permissions to role @role: @message', [
          '@role' => $roleId,
          '@message' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Grants specified permissions to a role.
   *
   * @param \Drupal\user\RoleInterface $role
   *   The role to grant permissions to.
   * @param array $permissions
   *   An array of permissions to grant. If empty, defaults will be used.
   *
   * @return \Drupal\user\RoleInterface
   *   The modified role entity.
   */
  public function grantPermissionsToRole(RoleInterface $role, array $permissions = []): RoleInterface {
    // Use default permissions if none provided.
    if (empty($permissions)) {
      $rolePermissions = $this->getDefaultRolePermissions();
      $permissions = $rolePermissions[$role->id()] ?? [];
    }

    // Grant each permission, checking if it already exists.
    foreach ($permissions as $permission) {
      if (!$role->hasPermission($permission)) {
        $role->grantPermission($permission);
      }
    }

    return $role;
  }

  /**
   * Gets the default role permissions mapping.
   *
   * Defines which permissions should be granted to which roles
   * when the image module is installed.
   *
   * @return array
   *   Array of role permissions keyed by role ID.
   */
  protected function getDefaultRolePermissions(): array {
    return [
      'content_author' => [
        'create image media',
        'edit own image media',
        'delete own image media',
      ],
      'content_editor' => [
        'edit any image media',
        'delete any image media',
      ],
    ];
  }

  /**
   * Configures editors to support image media.
   *
   * This method:
   * 1. Loads and configures filter formats to enable media embedding
   * 2. Updates associated CKEditor instances with media toolbar items.
   *
   * The order is important: filter formats must be configured before
   * their associated editors.
   */
  private function configureFilters(): void {
    $filterFormatIds = ['filtered_html', 'full_html'];

    // First, configure all filter formats.
    foreach ($filterFormatIds as $filterFormatId) {
      try {
        $filterFormat = $this->entityTypeManager->getStorage('filter_format')->load($filterFormatId);
        if ($filterFormat instanceof FilterFormatInterface) {
          $this->alterFilterFormat($filterFormat);
          $filterFormat->save();
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Failed to configure filter format @id: @message', [
          '@id' => $filterFormatId,
          '@message' => $e->getMessage(),
        ]);
      }
    }

    // Then, configure editors that use these filter formats.
    foreach ($filterFormatIds as $filterFormatId) {
      try {
        $editor = $this->entityTypeManager->getStorage('editor')->load($filterFormatId);
        if ($editor instanceof EditorInterface) {
          $this->alterEditor($editor);
          $editor->save();
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Failed to configure editor @id: @message', [
          '@id' => $filterFormatId,
          '@message' => $e->getMessage(),
        ]);
      }
    }
  }

  /**
   * Configures the editor to support image media.
   *
   * Updates the CKEditor configuration to include:
   * - Media toolbar button
   * - Media plugin settings.
   *
   * @param \Drupal\editor\EditorInterface $editor
   *   The editor entity to configure.
   */
  public function alterEditor(EditorInterface $editor): void {
    if ($editor->getEditor() !== 'ckeditor5') {
      return;
    }

    $settings = $editor->getSettings();
    $this->updateToolbarItems($settings);
    $editor->setSettings($settings);
  }

  /**
   * Configures the filter format to enable media embedding.
   *
   * Enables the media_embed filter which is required for
   * displaying embedded media in content.
   *
   * @param \Drupal\filter\FilterFormatInterface $filterFormat
   *   The filter format to configure.
   */
  public function alterFilterFormat(FilterFormatInterface $filterFormat): void {
    $filters = $filterFormat->get('filters');

    // Check if media_embed filter exists before enabling it.
    if (!isset($filters['media_embed'])) {
      $filters['media_embed']['status'] = TRUE;
      $filterFormat->set('filters', $filters);
    }
  }

  /**
   * Updates toolbar items to include media support.
   *
   * Adds the drupalMedia button to the CKEditor toolbar and
   * configures media plugin settings.
   *
   * @param array $settings
   *   The editor settings to update (passed by reference).
   */
  protected function updateToolbarItems(array &$settings): void {
    // Initialize toolbar items if not set.
    $settings['toolbar']['items'] ??= [];
    $items = &$settings['toolbar']['items'];

    // Add drupalMedia button if not already present.
    if (!in_array('drupalMedia', $items, TRUE)) {
      $items[] = 'drupalMedia';
    }

    // Configure media plugin settings.
    if (!isset($settings['plugins']['media_media']['allow_view_mode_override'])) {
      $settings['plugins']['media_media']['allow_view_mode_override'] = FALSE;
    }
  }

}
