<?php

namespace Drupal\Tests\acquia_cms_image\FunctionalJavascript;

use Drupal\Core\Field\FieldConfigInterface;
use Drupal\editor\Entity\Editor;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\filter\Entity\FilterFormat;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\node\NodeTypeInterface;
use Drupal\Tests\ckeditor5\Traits\CKEditor5TestTrait;
use Drupal\Tests\media\Traits\MediaTypeCreationTrait;

/**
 * Tests the configuration of CKEditor5 with the Acquia CMS Image module.
 *
 * @group acquia_cms_image
 */
class Ckeditor5ConfigurationTest extends WebDriverTestBase {

  use CKEditor5TestTrait;
  use MediaTypeCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'filter',
    'editor',
    'ckeditor5',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests media button exists in ckeditor when module installed first.
   */
  public function testMediaButtonsWhenModuleInstalledFirst() {
    $this->container->get('module_installer')->install([
      'acquia_cms_image',
    ]);
    $this->createTextFormatAndEditors();
    $this->assertMediaButtonExistsInEditor();
  }

  /**
   * Tests media button exists in ckeditor when module installed afterward.
   */
  public function testMediaButtonWhenModuleInstalledAfterwards() {
    $this->createTextFormatAndEditors();
    $this->container->get('module_installer')->install([
      'acquia_cms_image',
    ]);
    $this->assertMediaButtonExistsInEditor();
  }

  /**
   * Tests that the media button exists in the CKEditor5 editor.
   */
  private function assertMediaButtonExistsInEditor(): void {
    $session = $this->getSession();

    $node = $this->drupalCreateContentType();
    $node_type = $node->id();
    $this->addBodyField($node);

    $account = $this->drupalCreateUser([
      "create $node_type content",
      "use text format full_html",
      "use text format filtered_html",
    ]);
    $account->save();
    $this->drupalLogin($account);

    $this->drupalGet("/node/add/$node_type");

    // Ensure that text format 'filtered_html' & 'full_html' exists.
    $formats = $session->evaluateScript('Object.keys(drupalSettings.editor.formats)');
    $this->assertSame(['filtered_html', 'full_html'], $formats);

    // Resize window, so that all Ckeditor plugins are displayed.
    $this->getSession()->getDriver()->resizeWindow(1920, 1080);
    $this->waitForEditor();
    $this->assertNotEmpty($this->getEditorButton('Insert Media'));
  }

  /**
   * Creates text formats and editors for testing.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  private function createTextFormatAndEditors(): void {
    $textFormats = [
      'filtered_html' => 'Filtered HTML',
      'full_html' => 'Full HTML',
    ];

    $i = 0;
    foreach ($textFormats as $format => $label) {
      FilterFormat::create([
        'format' => $format,
        'name' => $label,
        'weight' => $i,
      ])->save();

      Editor::create([
        'editor' => 'ckeditor5',
        'format' => $format,
      ])->save();

      $i++;
    }

  }

  /**
   * Adds the body field to a node type.
   *
   * @param \Drupal\node\NodeTypeInterface $type
   *   The node type to which the body field should be added.
   * @param string $label
   *   The label for the body field.
   *
   * @return \Drupal\Core\Field\FieldConfigInterface
   *   The body field configuration.
   */
  private function addBodyField(NodeTypeInterface $type, string $label = 'Body'): FieldConfigInterface {
    // Add or remove the body field, as needed.
    $field_storage = FieldStorageConfig::loadByName('node', 'body');
    $field = FieldConfig::loadByName('node', $type->id(), 'body');
    if (empty($field)) {
      $field = FieldConfig::create([
        'field_storage' => $field_storage,
        'bundle' => $type->id(),
        'label' => $label,
        'settings' => [
          'display_summary' => TRUE,
          'allowed_formats' => [],
        ],
      ]);
      $field->save();

      /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $display_repository */
      $display_repository = $this->container->get('entity_display.repository');

      // Assign widget settings for the default form mode.
      $display_repository->getFormDisplay('node', $type->id())
        ->setComponent('body', [
          'type' => 'text_textarea_with_summary',
        ])
        ->save();

      // Assign display settings for the 'default' and 'teaser' view modes.
      $display_repository->getViewDisplay('node', $type->id())
        ->setComponent('body', [
          'label' => 'hidden',
          'type' => 'text_default',
        ])
        ->save();

      // The teaser view mode is created by the Standard profile and therefore
      // might not exist.
      $view_modes = $display_repository->getViewModes('node');
      if (isset($view_modes['teaser'])) {
        $display_repository->getViewDisplay('node', $type->id(), 'teaser')
          ->setComponent('body', [
            'label' => 'hidden',
            'type' => 'text_summary_or_trimmed',
          ])
          ->save();
      }
    }
    return $field;
  }

}
