<?php

declare(strict_types=1);

namespace Drupal\Tests\acquia_cms_image\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\Role;

/**
 * Tests the permissions granted by the Acquia CMS Image module.
 *
 * @group acquia_cms_image
 */
class PermissionsTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
  }

  /**
   * Tests permissions are granted to role created before module install.
   *
   * @dataProvider rolesPermissionDataProvider
   */
  public function testGrantPermissionsOnInstall(string $roleId, array $expectedPermissions): void {
    // Create the role before the module is installed.
    Role::create([
      'id' => $roleId,
      'label' => $this->randomString(),
    ])->save();

    // Install the module which should grant permissions to the existing role.
    $this->container->get('module_installer')->install(['acquia_cms_image']);

    // Reload the role to ensure permissions are applied.
    $role = Role::load($roleId);
    foreach ($expectedPermissions as $permission) {
      $this->assertTrue($role->hasPermission($permission), "Role `{$role->id()}` should have permission: $permission");
    }

  }

  /**
   * Tests permissions are granted to role created after module install.
   *
   * @dataProvider rolesPermissionDataProvider
   */
  public function testGrantPermissionsOnRoleCreate(string $roleId, array $expectedPermissions): void {
    $this->container->get('module_installer')->install(['acquia_cms_image']);

    // Create the role after the module is installed.
    Role::create([
      'id' => $roleId,
      'label' => $this->randomString(),
    ])->save();

    // Reload the role to ensure permissions are applied.
    $role = Role::load($roleId);
    foreach ($expectedPermissions as $permission) {
      $this->assertTrue($role->hasPermission($permission), "Role `{$role->id()}` should have permission: $permission");
    }

  }

  /**
   * Data provider for roles and their expected permissions.
   *
   * @return array
   *   An array of roles and their expected permissions.
   */
  public static function rolesPermissionDataProvider(): array {
    return [
      [
        'content_author',
        [
          'create image media',
          'delete own image media',
          'edit own image media',
        ],
      ],
      [
        'content_editor',
        [
          'delete any image media',
          'edit any image media',
        ],
      ],
    ];
  }

}
