<?php

namespace Drupal\acquia_contenthub_translations;

use Drupal\acquia_contenthub\Client\ClientFactory;
use Drupal\acquia_contenthub\Libs\Common\EnsureContentHubClientTrait;
use Drupal\acquia_contenthub_translations\Helpers\SubscriberLanguagesTrait;
use Drupal\acquia_contenthub_translations\UndesiredLanguageRegistry\UndesiredLanguageRegistryInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Psr\Log\LoggerInterface;

/**
 * Service for filtering translatable entities based on language settings.
 */
class TranslatableEntityFilter {

  use SubscriberLanguagesTrait;
  use EnsureContentHubClientTrait;

  /**
   * Language Manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected LanguageManagerInterface $languageManager;

  /**
   * Undesired language registrar.
   *
   * @var \Drupal\acquia_contenthub_translations\UndesiredLanguageRegistry\UndesiredLanguageRegistryInterface
   */
  protected UndesiredLanguageRegistryInterface $registrar;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a TranslatableEntityFilter object.
   *
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   Language Manager.
   * @param \Drupal\acquia_contenthub_translations\UndesiredLanguageRegistry\UndesiredLanguageRegistryInterface $registrar
   *   Undesired language registrar.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\acquia_contenthub\Client\ClientFactory $client_factory
   *   The Content Hub Client factory service.
   */
  public function __construct(LanguageManagerInterface $language_manager, UndesiredLanguageRegistryInterface $registrar, LoggerInterface $logger, ClientFactory $client_factory) {
    $this->languageManager = $language_manager;
    $this->registrar = $registrar;
    $this->logger = $logger;
    $this->clientFactory = $client_factory;
  }

  /**
   * Filters entity UUIDs based on subscriber languages.
   *
   * @param array $untracked_uuids
   *   Untracked UUIDs to check.
   * @param array $uuids
   *   Uuids to filter out.
   *
   * @return array
   *   Filtered out uuids based on subscriber languages.
   *
   * @throws \Exception
   */
  public function filterEntityUuidsBySubscriberLanguages(array $untracked_uuids, array $uuids): array {
    $client = $this->getClient();
    $untracked_cdf_document = $client->getEntities($untracked_uuids);
    $enabled_languages = $this->getOriginalEnabledLanguages($this->languageManager, $this->registrar);
    $deletable_uuids = [];
    $logs = [];

    foreach ($untracked_cdf_document->getEntities() as $untracked_cdf) {
      $languages = $untracked_cdf->getMetadata()['languages'] ?? [$untracked_cdf->getMetadata()['default_language']];
      if (empty(array_intersect($languages, $enabled_languages))) {
        $deletable_uuids[] = $untracked_cdf->getUuid();
        $logs[] = $untracked_cdf->getAttribute('entity_type')->getValue()[LanguageInterface::LANGCODE_NOT_SPECIFIED] . ' : ' . $untracked_cdf->getUuid();
      }
    }

    if (empty($deletable_uuids)) {
      return $uuids;
    }

    $this->logger->info('These entities ({entities}) were not added to import queue as these are in foreign languages.',
      ['entities' => implode(', ', $logs)]
    );

    return array_diff($uuids, $deletable_uuids);
  }

}
