<?php

namespace Drupal\acquia_contenthub\Libs\InterestList;

/**
 * Defines behaviour for interest list storage.
 */
interface InterestListStorageInterface {

  public const INTEREST_LIST_PAGINATION_SIZE = 3000;

  /**
   * Provides interest list if cached otherwise fetches from CH service.
   *
   * @param string $webhook_uuid
   *   Webhook uuid.
   * @param string $site_role
   *   Site role.
   * @param array $query
   *   Query params. Accepts disable_syndication, size, from and uuids.
   *
   * @code
   *   disable_syndication: boolean.
   *     Filter for disabled entities.
   *     If set to true, only disabled entities will be returned.
   *     If set to false, only enabled entities will be returned.
   *     If not set, all the entities will be returned.
   *   size: integer.
   *     Size of the interest list items to return.
   *     Max is 3000. Min is 1.
   *   from: integer.
   *     This is the offset value.
   *     Must be positive integer.
   *   uuids: array.
   *     Max 3000 uuids can be passed.
   * @endcode
   *
   * @return array
   *   An associate array keyed by the entity uuid.
   *
   * @throws \Drupal\acquia_contenthub\Exception\ContentHubClientException
   * @throws \Drupal\acquia_contenthub\Exception\ContentHubException
   */
  public function getInterestList(string $webhook_uuid, string $site_role, array $query = []): array;

  /**
   * Cleans up cached interest list.
   */
  public function resetCache(): void;

  /**
   * Checks if a single entity is in the interest list.
   *
   * @param string $webhook_uuid
   *   Webhook uuid.
   * @param string $entity_uuid
   *   Entity uuid to check.
   * @param string $site_role
   *   Site role.
   *
   * @return bool
   *   TRUE if entity is in interest list, FALSE otherwise.
   */
  public function isEntityInInterestList(string $webhook_uuid, string $entity_uuid, string $site_role): bool;

  /**
   * Checks if a single entity is disabled.
   *
   * @param string $webhook_uuid
   *   Webhook uuid.
   * @param string $entity_uuid
   *   Entity uuid to check.
   * @param string $site_role
   *   Site role.
   *
   * @return bool
   *   TRUE if entity is disabled, FALSE otherwise.
   */
  public function isEntityDisabled(string $webhook_uuid, string $entity_uuid, string $site_role): bool;

  /**
   * Filters uuids of disabled entities.
   *
   * @param string $webhook_uuid
   *   Webhook uuid.
   * @param array $uuids
   *   Source uuids.
   * @param string $site_role
   *   Site role.
   *
   * @return array
   *   List of entity uuids without disabled entity uuids.
   */
  public function filterDisabledEntities(string $webhook_uuid, array $uuids, string $site_role): array;

}
