<?php

declare(strict_types=1);

namespace Drupal\acquia_dam\Enum;

use Drupal\media\MediaInterface;

/**
 * Enum representing the status of an asset in DAM.
 */
enum AssetStatus: string {
  case RELEASED = 'Released';
  case EXPIRED = 'Expired';
  case UNRELEASED = 'Unreleased';
  case DELETED = 'Deleted';

  /**
   * Return asset status value.
   */
  public function getValue(): string {
    return $this->value;
  }

  /**
   * Check if the asset status is in sync with the media item.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media item to check against.
   *
   * @return bool
   *   TRUE if the asset status is in sync with media item, FALSE otherwise.
   */
  public function isInSync(MediaInterface $media): bool {
    return match ($this) {
      self::RELEASED => $media->isPublished(),
      self::EXPIRED, self::UNRELEASED, self::DELETED => !$media->isPublished(),
    };
  }

  /**
   * Determine asset status based on asset data and returns asset status enum.
   *
   * @param array $asset_data
   *   The asset data retrieved from the DAM.
   *
   * @return \Drupal\acquia_dam\Enum\AssetStatus
   *   Returns the asset status enum based on the asset data.
   */
  public static function getStatusFromAssetData(array $asset_data): self {
    // Determine the asset status based on its release and expiration dates.
    // If the asset is released and not expired, return 'Released'.
    if ($asset_data['released_and_not_expired']) {
      return self::RELEASED;
    }

    // If the asset does not have the release date, return 'Unreleased'.
    if ($asset_data['security']['release_date'] === NULL) {
      return self::UNRELEASED;
    }

    // If the asset has a release date but is expired, return 'Expired'.
    return self::EXPIRED;
  }

}
