<?php

declare(strict_types=1);

namespace Drupal\acquia_dam\Hook;

use Drupal\acquia_dam\Entity\MediaItemField;
use Drupal\acquia_dam\Plugin\media\acquia_dam\AssetMediaSourceManager;
use Drupal\canvas\Hook\ShapeMatchingHooks;
use Drupal\canvas\PropExpressions\StructuredData\FieldPropExpression;
use Drupal\canvas\PropExpressions\StructuredData\FieldTypeObjectPropsExpression;
use Drupal\canvas\PropExpressions\StructuredData\FieldTypePropExpression;
use Drupal\canvas\PropExpressions\StructuredData\ReferenceFieldTypePropExpression;
use Drupal\canvas\PropShape\CandidateStorablePropShape;
use Drupal\canvas\TypedData\BetterEntityDataDefinition;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\OrderAfter;

/**
 * @file
 * Hook implementations for acquia_dam & canvas integration.
 */
class CanvasHooks {

  /**
   * Construct a new CanvasHooks object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\acquia_dam\Plugin\media\acquia_dam\AssetMediaSourceManager $assetMediaSourceManager
   *   The asset media source manager.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly AssetMediaSourceManager $assetMediaSourceManager,
  ) {}

  /**
   * Mapping of JSON schema $ref values to media source plugin IDs.
   *
   * This is used to determine which media source plugins to add to the list of
   * allowed media types for a given storable prop shape.
   *
   * @see self::acquiaDamStoragePropShapeAlter()
   */
  const SCHEMA_TO_MEDIA_SOURCE_PLUGIN = [
    // @see \Drupal\media\Plugin\media\Source\Image
    'json-schema-definitions://canvas.module/image' => 'acquia_dam_asset:image',
  ];

  /**
   * Implements hook_storage_prop_shape_alter().
   *
   * Overrides (extends, really) Canvas' default choice of using the `image` media
   * type, to also allow \Drupal\acquia_dam\Plugin\media\acquia_dam\Image.
   *
   * @see \Drupal\canvas\Hook\ShapeMatchingHooks::mediaLibraryStoragePropShapeAlter()
   * @see \Drupal\media\Plugin\media\Source\Image
   * @see \Drupal\acquia_dam\Plugin\media\acquia_dam\Image
   */
  #[Hook('canvas_storable_prop_shape_alter',
    order: new OrderAfter(
      classesAndMethods: [
        [ShapeMatchingHooks::class, 'mediaLibraryStorablePropShapeAlter'],
      ]
    )
  )]
  public function acquiaDamStoragePropShapeAlter(CandidateStorablePropShape $storable_prop_shape): void {
    if (
      $storable_prop_shape->shape->schema['type'] === 'object'
      && isset($storable_prop_shape->shape->schema['$ref'])
      && array_key_exists($storable_prop_shape->shape->schema['$ref'], self::SCHEMA_TO_MEDIA_SOURCE_PLUGIN)
    ) {
      if ($storable_prop_shape->fieldWidget !== 'media_library_widget') {
        throw new \LogicException('This MUST run after \Drupal\canvas\Hook\ShapeMatchingHooks::mediaLibraryStoragePropShapeAlter(), because it builds on top of what did logic did.');
      }

      // Verifying there's an array of target bundles and each has an array of
      // media source field names being referenced.
      $media_type_ids = &$storable_prop_shape->fieldInstanceSettings['handler_settings']['target_bundles'];
      $field_names_per_media_type_id = $storable_prop_shape->fieldTypeProp->objectPropsToFieldTypeProps['src']->referenced->fieldName;

      // Upcast to array if there's only one media type ID or field name.
      if (is_string($field_names_per_media_type_id)) {
        assert(count($media_type_ids) === 1);
        $field_names_per_media_type_id = [
          reset($media_type_ids) => $field_names_per_media_type_id,
        ];
      }

      assert(is_array($field_names_per_media_type_id));
      assert(array_keys($field_names_per_media_type_id) === array_keys($media_type_ids));
      assert(array_keys($field_names_per_media_type_id) === array_values($media_type_ids));

      $media_source_plugin = self::SCHEMA_TO_MEDIA_SOURCE_PLUGIN[$storable_prop_shape->shape->schema['$ref']];

      // Get all Acquia DAM media types that use the given source plugin.
      $acquia_dam_media_types = $this->getMediaTypesForSourcePlugins($media_source_plugin);

      // Return an array of all property names for the given media source plugin.
      $property_map = $this->getPropertiesForMediaSource($media_source_plugin);

      // Prepare property name arrays for each property in the map.
      $prop_names_per_field_name = [];
      foreach (array_keys($property_map) as $prop) {
        $prop_names_per_field_name[$prop] = $this->getMappedPropNameFromStorablePropShape($storable_prop_shape, $prop);
        // If the prop name is NULL, it means the storable prop shape doesn't
        // define it. This is unexpected because the prop shape must have
        // defined all properties.
        assert(!empty($prop_names_per_field_name[$prop]), 'The prop name for each field must be defined.');
        if (is_string($prop_names_per_field_name[$prop])) {
          $prop_names_per_field_name[$prop] = $this->upcastPropNameToArray($prop_names_per_field_name[$prop], $media_type_ids, $field_names_per_media_type_id);
        }
      }

      // Add each Acquia DAM media type to the list of allowed bundles and
      // associate its source field_name with the corresponding prop names for
      // the media source.
      foreach ($acquia_dam_media_types as $bundle => $media_type) {
        $source_plugin_id = $media_type->getSource()->getPluginDefinition()['id'] ?? '';
        assert(is_string($source_plugin_id) && !empty($source_plugin_id), 'Media source plugin ID must be a non-empty string.');
        $fieldName = MediaItemField::getFieldName($source_plugin_id);
        $media_type_ids[$bundle] = $bundle;
        $field_names_per_media_type_id[$bundle] = $fieldName;
        foreach ($property_map as $prop => $mapped_name) {
          $prop_names_per_field_name[$prop][$fieldName] = $mapped_name;
        }
      }

      // Target bundles must be sorted.
      // @see \Drupal\canvas\TypedData\BetterEntityDataDefinition::create()
      ksort($media_type_ids);
      // Field mapping must match the sort order.
      ksort($field_names_per_media_type_id);

      // Property names must match the sort order of field names.
      // This ensures that when multiple media types are targeted, the property
      // names are aligned with their respective field names.
      // @see \Drupal\canvas\PropExpressions\StructuredData\FieldPropExpression::__construct()
      foreach (array_keys($property_map) as $prop) {
        $prop_names_per_field_name[$prop] = $this->sortPropNamesByFieldOrder($prop_names_per_field_name[$prop], $field_names_per_media_type_id);
      }

      // Now, update the expression with property mappings created above.
      // Each property ('src', 'alt', 'width', 'height') is mapped to its field
      // and prop names, ensuring alignment with the sorted field names and
      // media bundles.
      // @see \Drupal\canvas\Hook\ShapeMatchingHooks::mediaLibraryStoragePropShapeAlter()
      $storable_prop_shape->fieldTypeProp = $this->buildFieldTypeObjectPropsExpression(
        $property_map,
        $media_type_ids,
        $field_names_per_media_type_id,
        $prop_names_per_field_name
      );
    }
  }

  /**
   * Get all MediaTypes that use the given source plugin.
   *
   * @param string $media_source_plugin
   *   The media source plugin ID.
   *
   * @return \Drupal\media\MediaTypeInterface[]
   *   An array of MediaType entities, keyed by their machine name.
   */
  private function getMediaTypesForSourcePlugins(string $media_source_plugin): array {
    $media_types = $this->entityTypeManager->getStorage('media_type')
      ->loadByProperties(['source' => $media_source_plugin]);
    ksort($media_types);
    return $media_types;
  }

  /**
   * Upcast a property name to an array mapping field names to property names.
   *
   * If there's only one media type ID, the returned array will contain a single
   * entry mapping the single field name to the given property name.
   * If there are multiple media type IDs, the returned array will contain an
   * entry for each field name, all mapping to the same given property name.
   *
   * @param string $prop_name
   *   The property name to upcast.
   * @param array $media_type_ids
   *   The media type IDs targeted by the field instance.
   * @param array $field_names_per_media_type_id
   *   An array mapping media type IDs to their respective field names.
   */
  private function upcastPropNameToArray(string $prop_name, array $media_type_ids, array $field_names_per_media_type_id): array {
    return count($media_type_ids) === 1
      ? [reset($field_names_per_media_type_id) => $prop_name]
      : array_fill_keys(array_values($field_names_per_media_type_id), $prop_name);
  }

  /**
   * Sort property names by the order of their corresponding field names.
   *
   * This ensures that property names are aligned with the order of field names
   * for each media type, which is important for consistent mapping and usage.
   *
   * @param array $prop_names_per_field_name
   *   An array mapping field names to property names.
   * @param array $field_names_per_media_type_id
   *   An array mapping media type IDs to their respective field names.
   */
  private function sortPropNamesByFieldOrder(array $prop_names_per_field_name, array $field_names_per_media_type_id): array {
    $unique_field_name_per_media = array_unique(array_values($field_names_per_media_type_id));
    $prop_name_per_field_sort = function($a, $b) use ($unique_field_name_per_media) {
      $posA = array_search($a, $unique_field_name_per_media);
      $posB = array_search($b, $unique_field_name_per_media);
      return $posA - $posB;
    };
    uksort($prop_names_per_field_name, $prop_name_per_field_sort);
    return $prop_names_per_field_name;
  }



  /**
   * Get the mapping of property names for a given media source plugin.
   *
   * This defines how properties like 'src', 'alt', 'width', and 'height' map
   * to field names in the media entity for different media source plugins.
   *
   * @param string $media_source_plugin
   *   The media source plugin ID.
   */
  private function getPropertiesForMediaSource(string $media_source_plugin): array {
    // Extract the plugin type from the full plugin ID.
    $plugin_type = str_contains($media_source_plugin, ':')
      ? explode(':', $media_source_plugin)[1]
      : $media_source_plugin;

    $plugin_instance = $this->assetMediaSourceManager->createInstance($plugin_type);
    if (method_exists($plugin_instance, 'getCanvasPropertyMappings')) {
      return $plugin_instance->getCanvasPropertyMappings();
    }

    return [];
  }

  /**
   * Build a FieldTypeObjectPropsExpression dynamically based on property mappings.
   *
   * @param array $property_map
   *   The property mappings from the media source plugin.
   * @param array $media_type_ids
   *   The media type IDs.
   * @param array $field_names_per_media_type_id
   *   An array mapping media type IDs to their respective field names.
   * @param array $prop_names_per_field_name
   *   An array mapping property names to field names.
   *
   * @return \Drupal\canvas\PropExpressions\StructuredData\FieldTypeObjectPropsExpression
   *   The dynamically built field type object props expression.
   */
  private function buildFieldTypeObjectPropsExpression(array $property_map, array $media_type_ids, array $field_names_per_media_type_id, array $prop_names_per_field_name): FieldTypeObjectPropsExpression {
    $props = [];

    // Dynamically build each property expression based on the property map.
    foreach (array_keys($property_map) as $prop) {
      $props[$prop] = new ReferenceFieldTypePropExpression(
        new FieldTypePropExpression('entity_reference', 'entity'),
        new FieldPropExpression(
          BetterEntityDataDefinition::create('media', array_values($media_type_ids)),
          $field_names_per_media_type_id,
          NULL,
          $prop_names_per_field_name[$prop]
        ),
      );
    }

    return new FieldTypeObjectPropsExpression('entity_reference', $props);
  }

  /**
   * Get the mapped property name for a given storable prop shape and property.
   *
   * @param \Drupal\canvas\PropShape\CandidateStorablePropShape $storable_prop_shape
   *   The candidate storable prop shape.
   * @param string $prop
   *   The property name to look up (e.g., 'src', 'alt', 'width', 'height').
   */
  private function getMappedPropNameFromStorablePropShape(CandidateStorablePropShape $storable_prop_shape, string $prop): string|array|NULL {
    // This separate method exists because the storable prop shape may have
    // different object properties depending on which prop needs to be extracted.
    // The implementation will be updated as needed.
    return $storable_prop_shape->fieldTypeProp->objectPropsToFieldTypeProps[$prop]->referenced->propName ?? NULL;
  }

}
