<?php

namespace Drupal\acquia_key_provider\Plugin\KeyProvider;

use Drupal\acquia_key_provider\FileReaderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyInterface;
use Drupal\key\Plugin\KeyPluginFormInterface;
use Drupal\key\Plugin\KeyProviderBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Adds a key provider that follows Acquia best practices for file based keys.
 *
 * @KeyProvider(
 *   id = "acquia_file",
 *   label = @Translation("Acquia file"),
 *   description = @Translation("The Acquia File key provider allows a key to be stored in a specialized path that exists outside of disaster recovery backups."),
 *   tags = {
 *     "file",
 *   },
 *   key_value = {
 *     "accepted" = FALSE,
 *     "required" = FALSE
 *   }
 * )
 */
class AcquiaFileKeyProvider extends KeyProviderBase implements KeyPluginFormInterface {

  /**
   * The file reader service.
   *
   * @var \Drupal\acquia_key_provider\FileReaderInterface
   */
  protected FileReaderInterface $fileReader;

  /**
   * The current Acquia application.
   *
   * @var string
   */
  protected string $application;

  /**
   * The current Acquia environment.
   *
   * @var string
   */
  protected string $environment;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->fileReader = $container->get('acquia_key_provider.file_reader');

    /** @var \Drupal\acquia_key_provider\EnvironmentReaderInterface $environment_reader */
    $environment_reader = $container->get('acquia_key_provider.environment_reader');
    $instance->application = $environment_reader->getEnv('AH_SITE_GROUP');
    $instance->environment = $environment_reader->getEnv('AH_SITE_ENVIRONMENT');
    return $instance;
  }

  /**
   * Gets the file key path for the provided key name.
   *
   * @param string $key_name
   *   The key name to build a file path for.
   *
   * @return string
   *   The absolute path to the provided key.
   */
  protected function getKeyFilePath(string $key_name): string {
    return "/mnt/gfs/$this->application.$this->environment/nobackup/$key_name";
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'file_name' => '',
      'base64_encoded' => FALSE,
      'strip_line_breaks' => FALSE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    if (empty($this->application) || empty($this->environment)) {
      $this->messenger()->addWarning($this->t('This does not appear to be an Acquia hosted application. If this is a local development environment, the "AH_SITE_GROUP" and "AH_SITE_ENVIRONMENT" environment variables may have to be temporarily created in order to proceed.'));
    }

    $form['file_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('File name'),
      '#description' => $this->t('This file should exist within the %path directory. For more information about the "nobackup" directory, see <a href="@link">Storing private information in the file system</a>.', [
        '%path' => (!empty($this->application) && !empty($this->environment)) ? $this->getKeyFilePath('') : '/mnt/gfs/[application].[environment]/nobackup/',
        '@link' => 'https://docs.acquia.com/acquia-cloud-platform/storing-private-information-file-system',
      ]),
      '#required' => TRUE,
      '#default_value' => $this->getConfiguration()['file_name'],
    ];

    $form['strip_line_breaks'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Strip trailing line breaks'),
      '#description' => $this->t('Check this to remove any trailing line breaks from the file. Leave unchecked if there is a chance that a line break could be a valid character in the key.'),
      '#default_value' => $this->getConfiguration()['strip_line_breaks'],
    ];

    // If this key type is for an encryption key.
    if ($form_state->getFormObject()->getEntity()->getKeyType()->getPluginDefinition()['group'] == 'encryption') {
      // Add an option to indicate that the value is stored Base64-encoded.
      $form['base64_encoded'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Base64-encoded'),
        '#description' => $this->t('Check this if the key in the file is Base64-encoded.'),
        '#default_value' => $this->getConfiguration()['base64_encoded'],
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $key_provider_settings = $form_state->getValues();

    $file = $this->getKeyFilePath($key_provider_settings['file_name']);

    // Does the file exist?
    if (!$this->fileReader->isFile($file)) {
      $form_state->setErrorByName('file_name', $this->t('The file %file does not exist.', ['%file' => $file]));
    }
    elseif ((!$this->fileReader->isReadable($file))) {
      $form_state->setErrorByName('file_name', $this->t('The file %file is not readable.', ['%file' => $file]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->setConfiguration($form_state->getValues());
  }

  /**
   * {@inheritdoc}
   */
  public function getKeyValue(KeyInterface $key) {
    $file = $this->getKeyFilePath($this->configuration['file_name']);

    // Make sure the file exists and is readable.
    if (!$this->fileReader->isFile($file) || !$this->fileReader->isReadable($file)) {
      return NULL;
    }

    $key_value = $this->fileReader->getContents($file);

    if (isset($this->configuration['strip_line_breaks']) && $this->configuration['strip_line_breaks']) {
      $key_value = rtrim($key_value, "\n\r");
    }

    if (isset($this->configuration['base64_encoded']) && $this->configuration['base64_encoded']) {
      $key_value = base64_decode($key_value);
    }

    return $key_value;
  }

}
