<?php

namespace Drupal\Tests\acquia_key_provider\Kernel;

use Drupal\acquia_key_provider\FileReaderInterface;
use Drupal\KernelTests\KernelTestBase;
use Drupal\acquia_key_provider\Plugin\KeyProvider\AcquiaFileKeyProvider;
use Drupal\key\Entity\Key;
use Drupal\key\Plugin\KeyPluginManager;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Provides test coverage for the acquia file key provider.
 *
 * @group acquia_key_provider
 */
class AcquiaKeyProviderTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'key',
    'acquia_key_provider',
  ];

  /**
   * The mock file reader service.
   *
   * @var \Drupal\acquia_key_provider\FileReaderInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected FileReaderInterface|MockObject $fileReader;

  /**
   * The key plugin manager service.
   *
   * @var \Drupal\key\Plugin\KeyPluginManager
   */
  protected KeyPluginManager $keyPluginManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->keyPluginManager = $this->container->get('plugin.manager.key.key_provider');
    $this->fileReader = $this->createMock(FileReaderInterface::class);
    $this->container->set('acquia_key_provider.file_reader', $this->fileReader);
  }

  /**
   * Test case for a key that does not exist.
   */
  public function testKeyNotFound() {
    $plugin = $this->keyPluginManager->createInstance('acquia_file', [
      'file_name' => 'test',
      'base64_encoded' => FALSE,
      'strip_line_breaks' => FALSE,
    ]);
    assert($plugin instanceof AcquiaFileKeyProvider);
    $this->fileReader->method('isFile')->willReturn(FALSE);
    static::assertNull($plugin->getKeyValue($this->createMock(Key::class)));
  }

  /**
   * Test case for a key that is not readable.
   */
  public function testKeyNotReadable() {
    $plugin = $this->keyPluginManager->createInstance('acquia_file', [
      'file_name' => 'test',
      'base64_encoded' => FALSE,
      'strip_line_breaks' => FALSE,
    ]);
    assert($plugin instanceof AcquiaFileKeyProvider);
    $this->fileReader->method('isFile')->willReturn(TRUE);
    $this->fileReader->method('isReadable')->willReturn(FALSE);
    static::assertNull($plugin->getKeyValue($this->createMock(Key::class)));
  }

  /**
   * Test case for a key that exists and is readable.
   */
  public function testKeyReadable() {
    $plugin = $this->keyPluginManager->createInstance('acquia_file', [
      'file_name' => 'test',
      'base64_encoded' => FALSE,
      'strip_line_breaks' => FALSE,
    ]);
    assert($plugin instanceof AcquiaFileKeyProvider);
    $this->fileReader->method('isFile')->willReturn(TRUE);
    $this->fileReader->method('isReadable')->willReturn(TRUE);
    $this->fileReader->method('getContents')->willReturn('Super Secret');
    static::assertSame('Super Secret', $plugin->getKeyValue($this->createMock(Key::class)));
  }

  /**
   * Test case for a key that should strip line breaks.
   */
  public function testStripLineBreaks() {
    $plugin = $this->keyPluginManager->createInstance('acquia_file', [
      'file_name' => 'test',
      'base64_encoded' => FALSE,
      'strip_line_breaks' => TRUE,
    ]);
    assert($plugin instanceof AcquiaFileKeyProvider);
    $this->fileReader->method('isFile')->willReturn(TRUE);
    $this->fileReader->method('isReadable')->willReturn(TRUE);
    $this->fileReader->method('getContents')->willReturn("Super Secret\n\n\n\r\r\n\r");
    static::assertSame('Super Secret', $plugin->getKeyValue($this->createMock(Key::class)));
  }

  /**
   * Test case for a base64 encoded key.
   */
  public function testBase64Encoded() {
    $plugin = $this->keyPluginManager->createInstance('acquia_file', [
      'file_name' => 'test',
      'base64_encoded' => TRUE,
      'strip_line_breaks' => FALSE,
    ]);
    assert($plugin instanceof AcquiaFileKeyProvider);
    $this->fileReader->method('isFile')->willReturn(TRUE);
    $this->fileReader->method('isReadable')->willReturn(TRUE);
    $this->fileReader->method('getContents')->willReturn("U3VwZXIgU2VjcmV0");
    static::assertSame('Super Secret', $plugin->getKeyValue($this->createMock(Key::class)));
  }

}
