export class HtmlCssCollector {
  constructor(isDebugMode = false) {
    this.isDebugMode = isDebugMode;
    this.tree = null;
    this.css = [];
    this.defaultStyles = {};
    this.disallowedTagNames = ['SCRIPT', 'STYLE', 'NOSCRIPT'];
  }

  /**
   * Collects HTML and CSS from various sources in priority order
   * @param {Object} layout - Redux layout state
   * @returns {Promise<{dom_tree: Object, css: Array, html: string, version: string, viewport: {w: number, h: number}}>}
   */
  async collectPageContent(layout) {
    // Primary source: Experience Builder iframe
    const iframeContent = await this.getContentFromIframe();
    let htmlElement = null;
    let pageHtml = iframeContent.html;
    let targetDocument = null;

    if (iframeContent.html && iframeContent.document) {
      // Use iframe content as primary source
      htmlElement = iframeContent.document.documentElement;
      targetDocument = iframeContent.document;

      if (this.isDebugMode) {
        console.log('Using Experience Builder iframe content as primary source');
      }
    } else {
      // Fallback to current document
      htmlElement = document.documentElement;
      targetDocument = document;
      pageHtml = document.documentElement.outerHTML;

      if (this.isDebugMode) {
        console.log('Using current document as fallback source');
      }
    }

    // Last resort: Use layout data if no HTML found
    if (!pageHtml && layout) {
      pageHtml = this.generateHtmlFromLayout(layout);
      if (this.isDebugMode) {
        console.warn('Using layout data as final fallback');
      }
    }

    if (!pageHtml) {
      throw new Error('No HTML content could be collected from any source');
    }

    // Get viewport size from the target document
    const viewport = this.getViewPortSize(htmlElement, targetDocument);
    if (!viewport.width || !viewport.height) {
      throw new Error('No viewport size found');
    }

    // Set default computed styles using the target document
    this.setDefaultComputedStyles(targetDocument);
    // Process DOM tree from the ORIGINAL element, not the cleaned one
    const domTree = await this.processTree(htmlElement, targetDocument);
    // Remove extension elements from the HTML element
    const cleanedElement = this.removeExtensionElements(htmlElement);
    const cleanedHtml = this.cleanUpText(cleanedElement.outerHTML);

    return {
      dom_tree: domTree,
      css: this.css,
      html: cleanedHtml,
      version: '4.1.3',
      viewport: { w: viewport.width, h: viewport.height }
    };
  }

  /**
   * Attempts to extract content from Experience Builder iframe - PRIMARY SOURCE
   * @returns {Promise<{html: string, css: string, document: Document|null}>}
   */
  async getContentFromIframe() {
    try {
      const xbIframe = document.querySelector('[data-xb-swap-active="true"][title="Preview"]');

      if (!xbIframe || !xbIframe.contentDocument) {
        if (this.isDebugMode) {
          console.warn('Experience Builder iframe not found or contentDocument not accessible');
        }
        return { html: '', css: '', document: null };
      }

      const iframeDoc = xbIframe.contentDocument;

      // Wait for iframe document to be ready
      if (iframeDoc.readyState !== 'complete') {
        await new Promise(resolve => {
          const checkReady = () => {
            if (iframeDoc.readyState === 'complete') {
              resolve();
            } else {
              setTimeout(checkReady, 100);
            }
          };
          checkReady();
        });
      }

      // Additional wait for stylesheets to load
      await new Promise(resolve => setTimeout(resolve, 3000));

      // Get complete document content including DOCTYPE
      let html = '';

      // Add DOCTYPE if present
      if (iframeDoc.doctype) {
        html += `<!DOCTYPE ${iframeDoc.doctype.name}`;
        if (iframeDoc.doctype.publicId) {
          html += ` PUBLIC "${iframeDoc.doctype.publicId}"`;
        }
        if (iframeDoc.doctype.systemId) {
          html += ` "${iframeDoc.doctype.systemId}"`;
        }
        html += '>\n';
      } else {
        // Fallback DOCTYPE
        html += '<!DOCTYPE html>\n';
      }

    // Add the complete HTML element
    html += iframeDoc.documentElement.outerHTML;

      // We don't collect CSS here anymore since we'll process styles during DOM tree processing
      return {
        html,
        css: '', // CSS will be processed later through computed styles
        document: iframeDoc
      };
    } catch (error) {
      if (this.isDebugMode) {
        console.warn('Error accessing Experience Builder iframe content:', error.message);
      }
      return { html: '', css: '', document: null };
    }
  }

  /**
   * Generates basic HTML from layout data as fallback
   * @param {Object} layout - Redux layout state
   * @returns {string}
   */
  generateHtmlFromLayout(layout) {
    if (this.isDebugMode) {
      console.warn('Using layout data as fallback - this may produce limited results');
    }

    return `<!DOCTYPE html>
<html>
  <head>
    <title>Drupal Experience Builder</title>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
  </head>
  <body>
    <div class="layout-container">
      ${JSON.stringify(layout, ['title', 'content', 'metadata'], 2)}
    </div>
  </body>
</html>`;
  }

  /**
   * Removes extension elements from HTML (simplified without monsido logic)
   * @param {HTMLElement} html - HTML element to clean
   * @returns {HTMLElement} - Cleaned HTML element
   */
  removeExtensionElements(html) {
    const htmlClone = html.cloneNode(true);

    // Remove any remaining disallowed tag elements
    this.disallowedTagNames.forEach(tagName => {
      const elements = htmlClone.querySelectorAll(tagName);
      if (elements) {
        elements.forEach(element => {
          element.remove();
        });
      }
    });

    return htmlClone;
  }

  /**
   * Gets viewport size from the given HTML element and document
   * @param {HTMLElement} html - The HTML element to get viewport size from
   * @param {Document} targetDocument - The document to get viewport from
   * @returns {{width: number, height: number}}
   */
  getViewPortSize(html, targetDocument = document) {
    try {
      const win = targetDocument.defaultView || window;
      const viewportWidth = win.visualViewport?.width || win.innerWidth || html.clientWidth || 1024;
      const viewportHeight = win.visualViewport?.height || win.innerHeight || html.clientHeight || 768;

      return {
        width: viewportWidth,
        height: viewportHeight
      };
    } catch (error) {
      if (this.isDebugMode) {
        console.warn('Error getting viewport size:', error);
      }
      return { width: 1024, height: 768 };
    }
  }

  /**
   * Cleans up text content by removing excessive whitespace
   * @param {string} text - The text content to clean
   * @returns {string}
   */
  cleanUpText(text) {
    try {
      return text.replaceAll(/ +/g, ' ');
    } catch (error) {
      if (this.isDebugMode) {
        console.warn('Error cleaning up text:', error);
      }
      return text;
    }
  }

  /**
   * Sets default computed styles for the document element
   * @param {Document} targetDocument - The document to get default styles from
   */
  setDefaultComputedStyles(targetDocument = document) {
    try {
      const documentElement = targetDocument.documentElement;
      const win = targetDocument.defaultView;

      if (!win) {
        if (this.isDebugMode) {
          console.warn('No window context available for targetDocument');
        }
        // Fallback to main window
        this.defaultStyles = this.getStylesAsRecord(documentElement, document);
      } else {
        this.defaultStyles = this.getStylesAsRecord(documentElement, targetDocument);
      }

      if (Object.keys(this.defaultStyles).length === 0) {
        if (this.isDebugMode) {
          console.warn('No default styles collected, this may cause issues');
        }
      }

      const defaultStylesString = this.collectStyles(this.defaultStyles);
      const defaultCssRule = `${defaultStylesString}`;
      this.css.push(defaultCssRule);
    }
    catch (error) {
      if (this.isDebugMode) {
        console.error('Error setting default computed styles:', error);
      }
    }
  }

  /**
   * Processes the DOM tree recursively
   * @param {HTMLElement | ShadowRoot} el - Element to process
   * @param {Document} targetDocument - The document to get styles from
   * @returns {Promise<Object>} - Tree structure
   */
  async processTree(el, targetDocument = document) {
    return new Promise(async (resolve) => {
      setTimeout(async () => {
        const data = {};

        if (el.nodeType !== 11) { // not a shadowRoot
          data.tn = el.tagName.toUpperCase();
          data.ci = this.processStyles(el, targetDocument);
          data.a = this.getAttributesList(el);

          // Handle shadow root if present
          const shadowRoot = el.shadowRoot;
          if (shadowRoot) {
            data.sr = await this.processTree(shadowRoot, targetDocument);
          }
        }

        data.c = [];

        const nodes = Array.from(el.childNodes);
        const lastIndex = nodes.length - 1;

        if (!nodes.length && !data.c.length) {
          delete data.c;
          resolve(data);
        } else {
          for (let i = 0; i < nodes.length; i += 1) {
            const node = nodes[i];

            if (node.nodeType === 1) { // Element node
              const tagName = node.tagName.toUpperCase();
              if (this.disallowedTagNames.includes(tagName)) {
                // do nothing; cannot use 'continue' since need to go until resolve
              } else {
                const child = await this.processTree(node, targetDocument);
                data.c.push(child);
              }
            } else if (node.nodeType === 3) { // Text node
              const textContent = this.cleanUpText(node.textContent || '');
              if (textContent.trim()) {
                data.c.push({
                  t: textContent
                });
              }
            }

            if (lastIndex === i) {
              if (data.c && !data.c.length) {
                delete data.c;
              }
              resolve(data);
            }
          }
        }
      }, 0);
    });
  }

  /**
   * Processes styles for an element and returns CSS ID
   * @param {HTMLElement} el - Element to process
   * @param {Document} targetDocument - The document to get styles from
   * @returns {number|undefined} - CSS ID
   */
  processStyles (el, targetDocument) {
    if (this.isDebugMode) {
      console.log('Processing styles for element:', el.tagName, el);
    }

    const {styles, sameId} = this.collectUniqueStyles(el, targetDocument);

    if (this.isDebugMode) {
      console.log('Collected styles:', styles, 'Same ID:', sameId);
    }

    // No unique styles means the style is identical to the default
    let csId = 0;

    if (styles) {
      if (sameId === undefined) {
        csId = this.css.length;
        this.css.push(styles);
        if (this.isDebugMode) {
          console.log('Added new CSS rule at index:', csId, styles);
        }
      } else {
        csId = sameId;
        if (this.isDebugMode) {
          console.log('Reusing existing CSS rule at index:', csId);
        }
      }
    } else if (this.isDebugMode) {
      console.log('No styles collected for element:', el.tagName);
    }

    return csId;
  }

  /**
   * Gets all computed styles as a record object
   * @param {HTMLElement} el - Element to get styles from
   * @param {Document} targetDocument - The document to get styles from
   * @returns {Object} - Styles record
   */
  getStylesAsRecord(el, targetDocument = document) {
    try {
      const win = targetDocument.defaultView || window;
      const styleObj = win.getComputedStyle(el);

      if (!styleObj) {
        if (this.isDebugMode) {
          console.warn('getComputedStyle returned null for element:', el.tagName);
        }
        return {};
      }

      const result = {};
      for (let i = styleObj.length; i--;) {
        const nameString = styleObj[i];
        const value = styleObj.getPropertyValue(nameString);
        if (value) {
          result[nameString] = `${value};`;
        }
      }

      if (this.isDebugMode && Object.keys(result).length === 0) {
        console.warn('No computed styles found for element:', el.tagName, el);
      }

      return result;
    } catch (error) {
      if (this.isDebugMode) {
        console.error('Error getting computed styles:', error, 'for element:', el.tagName);
      }
      return {};
    }
  }

  /**
   * Collects unique styles for an element
   * @param {HTMLElement} el - Element to collect styles from
   * @param {Document} targetDocument - The document to get styles from
   * @returns {Object} - Unique styles and same ID if found
   */
  collectUniqueStyles(el, targetDocument) {
    const styles = this.collectStyles(this.getStylesAsRecord(el, targetDocument), this.defaultStyles);
    let sameId;

    if (styles.length) {
      const index = this.css.indexOf(styles);
      if (index !== -1) {
        sameId = index;
      }
    }
    return { styles, sameId };
  }

  /**
   * Collects styles as a string, optionally removing default styles
   * @param {Object} stylesObj - Styles object
   * @param {Object} [defaultStyles] - Default styles to remove
   * @returns {string} - Collected styles string (CSS properties without selector)
   */
  collectStyles(stylesObj, defaultStyles) {
    if (defaultStyles) {
      stylesObj = this.removeDefaultStyles(stylesObj, defaultStyles);
    }

    return Object.entries(stylesObj)
      .map(([key, value]) => `${this.escapeQuotation(key)}: ${this.escapeQuotation(value)}`)
      .join(' ');
  }

  /**
   * Removes default styles from styles object
   * @param {Object} stylesObj - Styles object
   * @param {Object} defaultObj - Default styles object
   * @returns {Object} - Filtered styles object
   */
  removeDefaultStyles(stylesObj, defaultObj) {
    const result = {};

    for (const k of Object.keys(stylesObj)) {
      if (stylesObj[k] !== defaultObj[k]) {
        result[k] = stylesObj[k];
      }
    }
    return result;
  }

  /**
   * Gets attributes list for an element
   * @param {HTMLElement} el - Element
   * @returns {Array} - Array of attribute pairs
   */
  getAttributesList(el) {
    const attrNames = Array.from(el.attributes);
    const result = attrNames.map(n => [n.nodeName, n.nodeValue || '']);
    return result;
  }

  /**
   * Escapes quotation marks in text
   * @param {string} text - Text to escape
   * @returns {string} - Escaped text
   */
  escapeQuotation(text) {
    return text.replaceAll(/"/g, '\\"');
  }

}