import { debugError } from '../utils/constants';

// API Error handling utility
class ApiError extends Error {
  constructor(message, statusCode, isRetryable = false, originalError = null, errorType = 'API_ERROR') {
    super(message);
    this.name = 'ApiError';
    this.statusCode = statusCode;
    this.isRetryable = isRetryable;
    this.originalError = originalError;
    this.errorType = errorType;
    this.timestamp = new Date().toISOString();
    this.preventFormSubmission = true; // Always prevent form submission on API errors
  }

  // Get user-friendly error details for UI display
  getErrorDetails() {
    return {
      message: this.message,
      statusCode: this.statusCode,
      errorType: this.errorType,
      timestamp: this.timestamp,
      isRetryable: this.isRetryable,
      preventFormSubmission: this.preventFormSubmission
    };
  }

  // Check if error should be displayed to user
  shouldDisplayToUser() {
    // Don't display network timeouts as prominently
    return this.errorType !== 'NETWORK_TIMEOUT';
  }

  // Get field-specific errors for form validation
  getFieldErrors() {
    const fieldErrors = {};

    if (this.statusCode === 400 && this.originalError?.details) {
      // Parse validation errors from API response
      const details = this.originalError.details;
      if (details.errors) {
        Object.keys(details.errors).forEach(field => {
          fieldErrors[field] = details.errors[field];
        });
      }
    } else if (this.statusCode === 401) {
      fieldErrors.api_key = 'Invalid API key. Please check your credentials.';
    } else if (this.statusCode === 403 || this.statusCode === 422) {
      fieldErrors.api_key = 'Access denied. Please verify your API key and permissions.';
    }

    return fieldErrors;
  }
}

// Status code to error message mapping
const statusMessages = {
  400: 'Invalid request. Please check your input and try again.',
  401: 'Authentication failed. Please check your API key.',
  403: 'Access denied. You may not have permission to access this resource.',
  404: 'The requested resource was not found.',
  422: 'Access to the Acquia Optimize API was denied. Contact your administrator for help.',
  429: 'Too many requests. Please wait before trying again.',
  500: 'Server error occurred. Please try again later.',
  502: 'Service temporarily unavailable. Please try again later.',
  503: 'Service temporarily unavailable. Please try again later.',
  504: 'Request timeout. Please try again later.',
};

// Error types for better categorization
const ERROR_TYPES = {
  VALIDATION_ERROR: 'VALIDATION_ERROR',
  AUTHENTICATION_ERROR: 'AUTHENTICATION_ERROR',
  AUTHORIZATION_ERROR: 'AUTHORIZATION_ERROR',
  NOT_FOUND_ERROR: 'NOT_FOUND_ERROR',
  RATE_LIMIT_ERROR: 'RATE_LIMIT_ERROR',
  SERVER_ERROR: 'SERVER_ERROR',
  NETWORK_ERROR: 'NETWORK_ERROR',
  TIMEOUT_ERROR: 'TIMEOUT_ERROR',
  UNKNOWN_ERROR: 'UNKNOWN_ERROR'
};

// Map status codes to error types
const statusCodeToErrorType = {
  400: ERROR_TYPES.VALIDATION_ERROR,
  401: ERROR_TYPES.AUTHENTICATION_ERROR,
  403: ERROR_TYPES.AUTHORIZATION_ERROR,
  404: ERROR_TYPES.NOT_FOUND_ERROR,
  422: ERROR_TYPES.AUTHORIZATION_ERROR,
  429: ERROR_TYPES.RATE_LIMIT_ERROR,
  500: ERROR_TYPES.SERVER_ERROR,
  502: ERROR_TYPES.SERVER_ERROR,
  503: ERROR_TYPES.SERVER_ERROR,
  504: ERROR_TYPES.TIMEOUT_ERROR,
};

// Timeout error codes
const TIMEOUT_CODES = {
  NETWORK_TIMEOUT: 'NETWORK_TIMEOUT',
  REQUEST_TIMEOUT: 'REQUEST_TIMEOUT',
  ABORT_TIMEOUT: 'ABORT_TIMEOUT'
};

// Retryable status codes
const RETRYABLE_STATUS_CODES = [429, 500, 502, 503, 504];

/**
 * Get user-friendly error message from status code
 * @param {number} statusCode - HTTP status code
 * @param {string} defaultMessage - Default message if status code not mapped
 * @returns {string} User-friendly error message
 */
const getErrorMessage = (statusCode, defaultMessage = 'An unexpected error occurred') => {
  return statusMessages[statusCode] || defaultMessage;
};

/**
 * Check if an error is retryable
 * @param {number|string} statusCode - HTTP status code or timeout code
 * @returns {boolean} Whether the error is retryable
 */
const isRetryableError = (statusCode) => {
  if (typeof statusCode === 'string') {
    // Handle timeout codes
    return Object.values(TIMEOUT_CODES).includes(statusCode);
  }
  return RETRYABLE_STATUS_CODES.includes(statusCode);
};

/**
 * Create a timeout promise that rejects after specified milliseconds
 * @param {number} ms - Timeout in milliseconds
 * @param {string} timeoutType - Type of timeout for error handling
 * @returns {Promise} Promise that rejects with timeout error
 */
const createTimeoutPromise = (ms, timeoutType = TIMEOUT_CODES.REQUEST_TIMEOUT) => {
  return new Promise((_, reject) => {
    setTimeout(() => {
      reject(new ApiError(
        'Request timeout. Please try again later.',
        timeoutType,
        true
      ));
    }, ms);
  });
};

/**
 * Enhanced fetch with timeout and comprehensive error handling
 * @param {string} url - Request URL
 * @param {Object} options - Fetch options
 * @param {number} timeout - Timeout in milliseconds (default: 30000)
 * @param {string} csrfToken - CSRF token for POST requests
 * @returns {Promise} Promise that resolves with response or rejects with ApiError
 */
const fetchWithErrorHandling = async (url, options = {}, timeout = 30000, csrfToken = null) => {
  const controller = new AbortController();
  const signal = controller.signal;

  const fetchOptions = {
    ...options,
    signal,
    credentials: 'same-origin', // Essential for Drupal session management
    mode: 'cors',
    headers: {
      'Content-Type': 'application/json',
      'Accept': 'application/json',
      'Cache-Control': 'no-cache, no-store, must-revalidate', // Stronger cache prevention
      'X-Requested-With': 'XMLHttpRequest', // Identifies as AJAX request
      'Pragma': 'no-cache', // Additional cache prevention
      'Expires': '0', // Prevent caching
      'Connection': 'keep-alive', // Maintain connection
      'User-Agent': navigator.userAgent, // Identify browser
      'Accept-Language': navigator.language || 'en-US', // Browser language
      'Accept-Encoding': 'gzip, deflate, br', // Compression support
      ...(csrfToken && ['POST'].includes(options.method?.toUpperCase()) && { 'X-CSRF-Token': csrfToken }), // CSRF token for state-changing operations
      ...options.headers,
    },
  };
  try {
    const response = await Promise.race([
      fetch(url, fetchOptions),
      createTimeoutPromise(timeout, TIMEOUT_CODES.REQUEST_TIMEOUT)
    ]);

    if (!response.ok) {
      const errorMessage = getErrorMessage(response.status);
      const isRetryable = isRetryableError(response.status);
      const errorType = statusCodeToErrorType[response.status] || ERROR_TYPES.UNKNOWN_ERROR;

      // Try to get additional error details from response body
      let errorDetails = null;
      try {
        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
          errorDetails = await response.json();
        }
        else {
          // Try to get text response for non-JSON errors
          const textResponse = await response.text();
          errorDetails = { rawResponse: textResponse };
          debugError(true, 'Non-JSON error response:', errorDetails);
        }
      } catch {
        // Ignore JSON parsing errors
      }

      throw new ApiError(errorMessage, response.status, isRetryable, {
        url,
        status: response.status,
        statusText: response.statusText,
        details: errorDetails
      }, errorType);
    }

    return response;
  } catch (error) {
    // Handle different types of errors
    if (error.name === 'AbortError') {
      throw new ApiError(
        'Request was cancelled. Please try again.',
        TIMEOUT_CODES.ABORT_TIMEOUT,
        true,
        error,
        ERROR_TYPES.TIMEOUT_ERROR
      );
    }

    if (error instanceof ApiError) {
      throw error;
    }

    // Network or other errors
    if (error.name === 'TypeError' && error.message.includes('fetch')) {
      throw new ApiError(
        'Network error. Please check your connection and try again.',
        TIMEOUT_CODES.NETWORK_TIMEOUT,
        true,
        error,
        ERROR_TYPES.NETWORK_ERROR
      );
    }

    throw new ApiError(
      'An unexpected error occurred. Please try again.',
      'UNKNOWN_ERROR',
      false,
      error,
      ERROR_TYPES.UNKNOWN_ERROR
    );
  }
};

/**
 * Retry wrapper for API calls
 * @param {Function} apiCall - Function that returns a Promise
 * @param {number} maxRetries - Maximum number of retries (default: 3)
 * @param {number} baseDelay - Base delay between retries in ms (default: 1000)
 * @returns {Promise} Promise that resolves with result or rejects with final error
 */
const withRetry = async (apiCall, maxRetries = 3, baseDelay = 1000) => {
  let lastError;

  for (let attempt = 0; attempt <= maxRetries; attempt++) {
    try {
      return await apiCall();
    } catch (error) {
      lastError = error;

      // Don't retry if it's not a retryable error or if we've reached max retries
      if (!error.isRetryable || attempt === maxRetries) {
        break;
      }

      // Exponential backoff with jitter
      const delay = baseDelay * Math.pow(2, attempt) + Math.random() * 1000;
      await new Promise(resolve => setTimeout(resolve, delay));
    }
  }

  throw lastError;
};

export const useApiService = (csrfToken = null) => {
  const fetchSettings = async () => {
    return withRetry(async () => {
      const response = await fetchWithErrorHandling(`/acquia-optimize/api/settings`, {
        method: 'GET',
      });

      return await response.json();
    });
  };

  const saveSettings = async (settings) => {
    return withRetry(async () => {
      const response = await fetchWithErrorHandling(`/acquia-optimize/api/settings`, {
        method: 'POST',
        body: JSON.stringify(settings)
      }, 30000, csrfToken);

      return await response.json();
    });
  };

  /**
   * Validate API connection using the provided credentials
   * This calls the PHP ApiClient::validateApiConnection() method
   * @param {string} apiKey - API key to validate
   * @param {string} apiUrl - API URL to validate
   * @returns {Promise} Promise that resolves with validation result or rejects with error
   */
  const validateApiConnection = async (apiKey, apiUrl) => {
    // Don't retry API connection validation - we want immediate feedback
    const response = await fetchWithErrorHandling(`/acquia-optimize/api/validate-connection`, {
      method: 'POST',
      body: JSON.stringify({
        api_key: apiKey,
        api_url: apiUrl
      })
    }, 30000, csrfToken);

    return await response.json();
  };

  const startQuickScan = async (requestBody) => {
    return withRetry(async () => {
      // Handle encoded_page specially - it needs to be a JSON string, but other fields should be normal
      const { encoded_page, ...otherFields } = requestBody;

      // Create the final request body with encoded_page properly stringified
      const finalRequestBody = {
        ...otherFields,
        encoded_page: JSON.stringify(encoded_page) // Convert object to JSON string
      };

      const response = await fetchWithErrorHandling(`/acquia-optimize/api/quick-scan`, {
        method: 'POST',
        body: JSON.stringify(finalRequestBody)
      }, 60000, csrfToken);

      return response;
    });
  };

  const checkScanStatus = async (scanId, isDebugMode) => {
    try {
      const response = await fetchWithErrorHandling(`/acquia-optimize/api/quick-scan/${scanId}`, {
        method: 'GET',
      }, 45000);

      const statusData = await response.json();
      return statusData;
    } catch (error) {
      if (error instanceof ApiError) {
        debugError(isDebugMode, 'Failed to check scan status:', error.statusCode, error.message);

        // For scan status checks, handle specific errors differently
        if (error.statusCode === 404) {
          // Scan might not be ready yet, return null to continue polling
          return null;
        } else if ([500, 502, 503].includes(error.statusCode)) {
          // Server errors during scan - throw to trigger retry at higher level
          throw error;
        } else {
          // Other errors should be propagated
          throw error;
        }
      }

      debugError(isDebugMode, 'Error checking scan status:', error);
      throw error; // Don't return null for unexpected errors
    }
  };

  return {
    fetchSettings,
    saveSettings,
    validateApiConnection,
    startQuickScan,
    checkScanStatus
  };
};

// Export error handling utilities
export {
  ApiError,
  statusMessages,
  TIMEOUT_CODES,
  ERROR_TYPES,
  getErrorMessage,
  isRetryableError,
  fetchWithErrorHandling,
  withRetry
};
