import React, { useState } from 'react';
import Button from '../Button';
import Footer from '../Footer';
import { ApiError, useApiService } from '../services/apiService';

const ApiKeyFormView = ({
  apiKey,
  setApiKey,
  apiUrl,
  setApiUrl,
  accessibilityGuideline,
  setAccessibilityGuideline,
  handleSubmit,
  isLoading,
  setShowApiKeyForm,
  helperText,
  canConfigure
}) => {
  const [errors, setErrors] = useState({});
  const [touched, setTouched] = useState({});
  const [apiErrors, setApiErrors] = useState({});
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isValidatingConnection, setIsValidatingConnection] = useState(false);

  // Initialize API service
  const { validateApiConnection } = useApiService();

  // Validation functions
  const validateApiKey = (value) => {
    if (!value || !value.trim()) {
      return 'API key is required';
    }

    if (value.length < 20) {
      return 'API Key appears to be too short. Please check your API key.';
    }

    // Check for suspicious characters that might indicate copy-paste errors
    if (/[<>"'\s]/.test(value)) {
      return 'API Key contains invalid characters. Please ensure you copied the key correctly.';
    }

    return null;
  };

  const validateApiUrl = (value) => {
    if (!value || !value.trim()) {
      return 'API URL is required';
    }

    // Basic URL validation
    try {
      const url = new URL(value);
      if (!['http:', 'https:'].includes(url.protocol)) {
        return 'API URL must use HTTP or HTTPS protocol';
      }
    } catch {
      return 'Please enter a valid URL (e.g., https://api.example.com/v1)';
    }

    return null;
  };

  // Field change handlers with validation
  const handleApiKeyChange = (e) => {
    const value = e.target.value;
    setApiKey(value);

    // Clear both validation and API errors when user starts typing
    if (apiErrors.api_key || apiErrors.general) {
      setApiErrors(prev => {
        const { api_key, general, ...rest } = prev;
        return rest;
      });
    }

    // Also clear validation errors for this field
    if (errors.api_key) {
      setErrors(prev => ({
        ...prev,
        api_key: null
      }));
    }

    if (touched.api_key) {
      const error = validateApiKey(value);
      setErrors(prev => ({
        ...prev,
        api_key: error
      }));
    }
  };

  const handleApiUrlChange = (e) => {
    const value = e.target.value;
    setApiUrl(value);

    // Clear both validation and API errors when user starts typing
    if (apiErrors.api_url || apiErrors.general) {
      setApiErrors(prev => {
        const { api_url, general, ...rest } = prev;
        return rest;
      });
    }

    // Also clear validation errors for this field
    if (errors.api_url) {
      setErrors(prev => ({
        ...prev,
        api_url: null
      }));
    }

    if (touched.api_url) {
      const error = validateApiUrl(value);
      setErrors(prev => ({
        ...prev,
        api_url: error
      }));
    }
  };

  // Field blur handlers
  const handleFieldBlur = (fieldName, value) => {
    setTouched(prev => ({
      ...prev,
      [fieldName]: true
    }));

    let error = null;
    if (fieldName === 'api_key') {
      error = validateApiKey(value);
    } else if (fieldName === 'api_url') {
      error = validateApiUrl(value);
    }

    setErrors(prev => ({
      ...prev,      [fieldName]: error
    }));
  };

  // Helper function to get the combined error for a field
  const getFieldError = (fieldName) => {
    return errors[fieldName] || apiErrors[fieldName] || null;
  };

  // Helper function to check if field has any error
  const hasFieldError = (fieldName) => {
    return !!(errors[fieldName] || apiErrors[fieldName]);
  };

  // Helper function to check if form can be submitted
  const isFormValid = () => {
    const hasValidationErrors = Object.values(errors).some(error => error !== null);
    const hasApiErrors = Object.values(apiErrors).some(error => error !== null);
    const isValid = !hasValidationErrors && !hasApiErrors && !isSubmitting && !isValidatingConnection;

    // Debug logging to help track state
    if (process.env.NODE_ENV === 'development') {
      console.log('Form validation state:', {
        hasValidationErrors,
        hasApiErrors,
        isSubmitting,
        isValidatingConnection,
        isValid,
        errors,
        apiErrors
      });
    }

    return isValid;
  };

  // Helper function to clear all errors
  const clearAllErrors = () => {
    setErrors({});
    setApiErrors({});
  };  const handleFormSubmit = async (e) => {
    e.preventDefault();

    // Prevent double submission
    if (isSubmitting || isValidatingConnection) {
      return;
    }

    // Clear all previous errors at the start
    clearAllErrors();

    setIsSubmitting(true);

    try {
      // Validate all fields first
      const apiKeyError = validateApiKey(apiKey);
      const apiUrlError = validateApiUrl(apiUrl);

      const newErrors = {
        api_key: apiKeyError,
        api_url: apiUrlError
      };

      setErrors(newErrors);
      setTouched({
        api_key: true,
        api_url: true
      });

      // Check if there are any validation errors
      const hasValidationErrors = Object.values(newErrors).some(error => error !== null);

      if (hasValidationErrors) {
        // Focus first field with error
        const firstErrorField = Object.keys(newErrors).find(key => newErrors[key]);
        if (firstErrorField) {
          const element = document.getElementById(firstErrorField);
          if (element) {
            element.focus();
            element.scrollIntoView({ behavior: 'smooth', block: 'center' });
          }
        }
        return; // Don't submit if validation fails
      }

      // Validate API connection before submitting the form
      setIsValidatingConnection(true);

      try {
        const connectionResult = await validateApiConnection(apiKey, apiUrl);

        // Check if the connection validation failed
        if (connectionResult.status && connectionResult.status !== 200) {
          // Map API connection errors to form fields
          const connectionErrors = {};

          if (connectionResult.status === 401) {
            connectionErrors.api_key = 'Invalid API key. Please check your credentials.';
          } else if (connectionResult.status === 403 || connectionResult.status === 422) {
            connectionErrors.api_key = 'Access denied. Please verify your API key and permissions.';
          } else if (connectionResult.status === 400) {
            // Check if it's a URL-related error
            if (connectionResult.error && (
                connectionResult.error.includes('URL') ||
                connectionResult.error.includes('connect') ||
                connectionResult.error.includes('resolve host') ||
                connectionResult.error.includes('accessible')
              )) {
              connectionErrors.api_url = connectionResult.error;
            } else {
              connectionErrors.api_key = connectionResult.error || 'Invalid API key. Please check your credentials.';
            }
          } else if (connectionResult.status === 404) {
            connectionErrors.api_url = 'API endpoint not found. Please verify the API URL.';
          } else if (connectionResult.status === 408) {
            connectionErrors.general = 'Connection timeout. Please check your network connection and try again.';
          } else if (connectionResult.status === 0) {
            // Network error
            connectionErrors.general = 'Unable to connect to the API. Please check your network connection and API URL.';
          } else {
            connectionErrors.general = connectionResult.error || 'Failed to connect to the API. Please try again.';
          }

          setApiErrors(connectionErrors);

          // Focus the first field with an error (prefer field-specific errors over general)
          const firstErrorField = Object.keys(connectionErrors).find(key =>
            connectionErrors[key] && key !== 'general'
          );
          if (firstErrorField) {
            const element = document.getElementById(firstErrorField);
            if (element) {
              element.focus();
              element.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
          }

          return; // Don't submit if API connection fails
        }

        // Connection validation successful, proceed with form submission
        await handleSubmit(e);

      } catch (connectionError) {
        // Handle API connection validation errors
        console.error('API connection validation error:', connectionError);

        if (connectionError instanceof ApiError) {
          const fieldErrors = connectionError.getFieldErrors();

          if (Object.keys(fieldErrors).length > 0) {
            setApiErrors(fieldErrors);

            // Focus the first field with an error
            const firstErrorField = Object.keys(fieldErrors)[0];
            const element = document.getElementById(firstErrorField);
            if (element) {
              element.focus();
              element.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
          } else {
            // Check if it's a network-related error
            let errorMessage = connectionError.message;
            let targetField = 'general';

            if (errorMessage.includes('URL') ||
                errorMessage.includes('connect') ||
                errorMessage.includes('resolve') ||
                errorMessage.includes('Network error')) {
              targetField = 'api_url';
              errorMessage = 'Unable to connect to the API URL. Please verify the URL is correct and accessible.';
            }

            setApiErrors({ [targetField]: errorMessage });

            if (targetField !== 'general') {
              const element = document.getElementById(targetField);
              if (element) {
                element.focus();
                element.scrollIntoView({ behavior: 'smooth', block: 'center' });
              }
            }
          }
        } else {
          // Handle non-ApiError exceptions
          let errorMessage = 'Failed to validate API connection. Please try again.';
          let targetField = 'general';

          if (connectionError.message) {
            const msg = connectionError.message.toLowerCase();
            if (msg.includes('url') || msg.includes('connect') || msg.includes('resolve') || msg.includes('network')) {
              targetField = 'api_url';
              errorMessage = 'Unable to connect to the API URL. Please verify the URL is correct and accessible.';
            }
          }

          setApiErrors({ [targetField]: errorMessage });

          if (targetField !== 'general') {
            const element = document.getElementById(targetField);
            if (element) {
              element.focus();
              element.scrollIntoView({ behavior: 'smooth', block: 'center' });
            }
          }
        }

        return; // Don't submit if connection validation fails
      } finally {
        setIsValidatingConnection(false);
      }

    } catch (error) {
      // Handle other form submission errors
      if (error instanceof ApiError) {
        const fieldErrors = error.getFieldErrors();

        if (Object.keys(fieldErrors).length > 0) {
          // Set field-specific errors
          setApiErrors(fieldErrors);

          // Focus the first field with an error
          const firstErrorField = Object.keys(fieldErrors)[0];
          const element = document.getElementById(firstErrorField);
          if (element) {
            element.focus();
            element.scrollIntoView({ behavior: 'smooth', block: 'center' });
          }
        } else {
          // Set general API error message
          setApiErrors({ general: error.message });
        }

        // Log error details for debugging
        console.error('API Error:', error.getErrorDetails());
      } else {
        // Handle non-API errors
        setApiErrors({ general: 'An unexpected error occurred. Please try again.' });
        console.error('Form submission error:', error);
      }
    } finally {
      setIsSubmitting(false);
      setIsValidatingConnection(false);
    }
  };
  // API Key configuration form - only shown if user has permission to configure
  if (!canConfigure) {
    return (
      <div className="ao-container">
        <div className="ao-message ao-message--error">
          You do not have permission to configure API settings.
        </div>
      </div>
    );
  }

  return (
    <div className="ao-container">
      <div className="ao-inner-container">
        <form onSubmit={handleFormSubmit} className="ao-form" noValidate>
          <div className="ao-form-group">
            <label
              htmlFor="api_key"
              className="ao-label"
            >
              API key <span className="ao-required">*</span>
              <span className="ao-info" data-tooltip="Enter your Acquia Optimize API key provided in your account dashboard">ⓘ</span>
            </label>
            <textarea
              id="api_key"
              value={apiKey}
              onChange={handleApiKeyChange}
              onBlur={(e) => handleFieldBlur('api_key', e.target.value)}
              placeholder="Enter API key here"
              required
              className={`ao-textarea ${hasFieldError('api_key') ? 'ao-textarea--error' : ''}`}
              aria-invalid={hasFieldError('api_key') ? 'true' : 'false'}
              aria-describedby={hasFieldError('api_key') ? 'api_key-error' : undefined}
            />
            {hasFieldError('api_key') && (
              <div
                id="api_key-error"
                className="ao-error-message"
                role="alert"
              >
                {getFieldError('api_key')}
              </div>
            )}
          </div>

          <div className="ao-form-group">
            <label
              htmlFor="api_url"
              className="ao-label"
            >
              API URL <span className="ao-required">*</span>
              <span className="ao-info" data-tooltip="Enter the URL endpoint for API calls (e.g., https://api.example.com/v1)">ⓘ</span>
            </label>
            <input
              id="api_url"
              type="text"
              value={apiUrl}
              onChange={handleApiUrlChange}
              onBlur={(e) => handleFieldBlur('api_url', e.target.value)}
              placeholder="Enter API url here"
              required
              className={`ao-input ${hasFieldError('api_url') ? 'ao-input--error' : ''}`}
              aria-invalid={hasFieldError('api_url') ? 'true' : 'false'}
              aria-describedby={hasFieldError('api_url') ? 'api_url-error' : undefined}
            />
            {hasFieldError('api_url') && (
              <div
                id="api_url-error"
                className="ao-error-message"
                role="alert"
              >
                {getFieldError('api_url')}
              </div>
            )}
          </div>

          <div className="ao-form-group">
            <label
              htmlFor="accessibility"
              className="ao-label"
            >
              Accessibility Guideline <span className="ao-required">*</span>
              <span className="ao-info" data-tooltip="Choose which WCAG accessibility standard to test against">ⓘ</span>
            </label>
            <div className="ao-select-container">
              <select
                id="accessibility"
                value={accessibilityGuideline}
                onChange={(e) => setAccessibilityGuideline(e.target.value)}
                required
                className="ao-select"
              >
                <option value="WCAG2-A">WCAG 2.0 A</option>
                <option value="WCAG2-AA">WCAG 2.0 AA</option>
                <option value="WCAG2-AAA">WCAG 2.0 AAA</option>
                <option value="WCAG21-A">WCAG 2.1 A</option>
                <option value="WCAG21-AA">WCAG 2.1 AA</option>
                <option value="WCAG21-AAA">WCAG 2.1 AAA</option>
                <option value="WCAG22-A">WCAG 2.2 A</option>
                <option value="WCAG22-AA">WCAG 2.2 AA</option>
                <option value="WCAG22-AAA">WCAG 2.2 AAA</option>
              </select>
              <div className="ao-select-arrow">▼</div>
            </div>
          </div>

          {helperText && (
            <div className={`ao-message ${helperText.includes('Error') ? 'ao-message--error' : 'ao-message--success'}`}>
              {helperText}
            </div>
          )}

          {apiErrors.general && (
            <div className="ao-message ao-message--error" role="alert">
              {apiErrors.general}
            </div>
          )}

          {(isLoading || isSubmitting || isValidatingConnection) ? (
            <div className="ao-loading-center">
              <div className="ao-spinner ao-spinner-form"></div>
              {isValidatingConnection && (
                <p className="ao-loading-text">Validating API connection...</p>
              )}
            </div>
          ) : (
            <div className="ao-form-actions">
              <button
                type="button"
                onClick={() => setShowApiKeyForm(false)}
                className="ao-button-secondary"
                disabled={isSubmitting || isValidatingConnection}
              >
                Cancel
              </button>

              <Button
                type="submit"
                disabled={!isFormValid()}
              >
                {isValidatingConnection ? 'Validating...' : isSubmitting ? 'Adding...' : 'Add'}
              </Button>
            </div>
          )}

          <Footer />
        </form>
      </div>
    </div>
  );
};

export default ApiKeyFormView;
