# Contributing to Acquia Optimize

Thank you for your interest in contributing to the Acquia Optimize module! This guide will help you get started with development and contributions.

## Getting Started

### Prerequisites
- PHP 8.1+ (recommended)
- Node.js 16+ (for UI development)
- npm 8+ or yarn
- Drupal 10+ development environment
- Composer 2.0+
- Development tools: DDEV or similar (DDEV recommended)

### Local Development Setup

1. **Clone the repository**:
```bash
git clone git@git.drupal.org:project/acquia_optimize.git
cd acquia_optimize
```

2. **Install dependencies**:
```bash
composer install
```

3. **Set up your Drupal environment**:
```bash
ddev start
ddev composer require drupal/acquia_optimize
ddev drush en acquia_optimize
```

## Building the UI

The Acquia Optimize module includes a React-based UI for the Experience Builder integration. To build the UI:

```bash
cd ui
npm install
npm run build
```

For development with hot reloading:
```bash
npm run dev
```

This will compile the React components into the `ui/dist/bundle.js` file that gets loaded by Drupal.

## Development Workflow

1. **Make your code changes**
2. **Build the UI components** (if UI changes were made)
3. **Clear the Drupal cache**: `ddev drush cr`
4. **Test your changes** in both regular content editing and Experience Builder interface
5. **Run tests** to ensure no regressions
6. **Check coding standards** compliance

## Architecture Overview

Understanding the module structure will help with contributions:

- **Controller**: `src/Controller/XbAcquiaOptimizeController.php` - Handles AJAX requests and API interactions
- **Forms**: `src/Form/SettingsForm.php` - Configuration interface
- **Services**: `src/Utilities.php` - Core processing logic
- **API Client**: `src/ApiClient.php` and `src/ApiClientFactory.php` - External API integration
- **Templates**: `templates/` - Twig templates for result rendering
- **JavaScript**: `js/` - Frontend interactions
- **React UI**: `ui/` - Experience Builder integration components

## Testing

### Running Tests

**PHPUnit tests**:
```bash
ddev exec XDEBUG_MODE=coverage phpunit web/modules/contrib/acquia_optimize/tests
```

**Specific test groups**:
```bash
# Unit tests
ddev exec phpunit --group acquia-optimize-unit

# Functional tests
ddev exec phpunit --group acquia-optimize-functional

# Integration tests
ddev exec phpunit --group acquia-optimize-integration
```

### Test Coverage

The module includes comprehensive test coverage:
- Unit tests for utility functions and data processing
- Functional tests for form validation and API integration
- Integration tests for complete workflow testing
- Render tests for template output verification

### Writing Tests

When adding new features, please include:
- Unit tests for new utility functions
- Functional tests for new forms or controllers
- Integration tests for API interactions
- Mock data for testing API responses

## Coding Standards

### PHP Code
- Follow [Drupal coding standards](https://www.drupal.org/docs/develop/standards)
- Use PHP_CodeSniffer: `phpcs --standard=Drupal,DrupalPractice`
- Use PHP Code Beautifier: `phpcbf --standard=Drupal,DrupalPractice`

### JavaScript/React Code
- Follow React best practices
- Use ESLint for code quality
- Use Prettier for code formatting
- Follow modern JavaScript (ES6+) patterns

### CSS/SCSS
- Follow Drupal CSS coding standards
- Use BEM methodology for class naming
- Ensure responsive design principles

## API Integration

When working with API features:

1. **Use the ApiClientFactory**: Don't create direct API clients
2. **Handle errors gracefully**: Always check for API errors
3. **Add logging**: Use the logger service for debugging
4. **Test with mock data**: Use the provided mock responses
5. **Validate responses**: Always validate API response structure

## Submitting Changes

### Pull Request Process

1. **Create a feature branch** from `1.x`:
```bash
git checkout -b feature/your-feature-name
```

2. **Make your changes** following the coding standards

3. **Build the UI** (if applicable):
```bash
cd ui && npm run build
```

4. **Run tests** and ensure they pass:
```bash
ddev exec phpunit web/modules/contrib/acquia_optimize/tests
```

5. **Check coding standards**:
```bash
phpcs --standard=Drupal,DrupalPractice src/
```

6. **Commit your changes** with clear commit messages

7. **Push to your fork** and submit a pull request

### Pull Request Guidelines

- **Clear description**: Explain what your changes do and why
- **Link to issues**: Reference any related issue numbers
- **Screenshots**: Include screenshots for UI changes
- **Test coverage**: Ensure new code is covered by tests
- **Documentation**: Update documentation if needed

### Code Review Process

All contributions go through code review:
- Automated tests must pass
- Code must follow Drupal standards
- Security review for sensitive changes
- UI/UX review for interface changes
- Performance review for optimization changes

## Common Development Tasks

### UI Changes
1. Modify React components in `ui/src/`
2. Update CSS/SCSS styles
3. Build and test in Experience Builder
4. Ensure responsive design

## Debugging

### Debug Mode
Enable debug mode in the module configuration for detailed logging:
- API request/response logging
- Processing step tracking
- Error details and stack traces

### Common Issues
- **API Connection**: Check network connectivity and credentials
- **Cache Issues**: Clear Drupal cache after changes
- **Permission Errors**: Verify user permissions

## Resources

- [Drupal Coding Standards](https://www.drupal.org/docs/develop/standards)
- [React Documentation](https://reactjs.org/docs)
- [Experience Builder Documentation](https://www.drupal.org/project/experience_builder)
- [Drupal Testing Guide](https://www.drupal.org/docs/testing)

## Getting Help

- **Issue Queue**: [Project issue queue](https://www.drupal.org/project/issues/acquia_optimize)
- **Documentation**: Check the README.md for usage instructions
- **Community**: Join Drupal Slack #acquia-optimize channel
- **Maintainers**: Contact the module maintainers for complex issues
