<?php

declare(strict_types=1);

namespace Drupal\acquia_optimize\Form;

use Drupal\acquia_optimize\ApiClientFactory;
use Drupal\acquia_optimize\Utilities;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Acquia Optimize.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Constructor.
   *
   * @param \Drupal\acquia_optimize\ApiClientFactory $apiClientFactory
   *   The API client service.
   */
  public function __construct(
    private readonly ApiClientFactory $apiClientFactory,
  ) {}

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'acquia_optimize.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('acquia_optimize.api_client_factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'acquia_optimize_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('acquia_optimize.settings');

    $form['api_key'] = [
      '#type' => 'textarea',
      '#title' => $this->t('API Key'),
      '#default_value' => $config->get('api_key'),
      '#description' => $this->t('Enter your Acquia Optimize API key. This key is required to connect to the service.'),
      '#required' => TRUE,
    ];

    $form['api_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API URL'),
      '#default_value' => $config->get('api_url'),
      '#description' => $this->t('Enter the API URL for the Acquia Optimize service. Must use HTTPS.'),
      '#required' => TRUE,
      '#element_validate' => [[$this, 'validateApiUrlElement']],
    ];

    $guidelines = Utilities::ACCESSIBILITY_GUIDELINES;
    $form['accessibility'] = [
      '#type' => 'select',
      '#title' => $this->t('Accessibility Guideline'),
      '#default_value' => $config->get('accessibility'),
      '#options' => array_combine($guidelines, $guidelines),
    ];
    $form['debug_mode'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug Mode'),
      '#description' => $this->t('Enable debug logging in the JavaScript console.'),
      '#default_value' => $this->config('acquia_optimize.settings')->get('debug_mode') ?? FALSE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    // Only test API connection if basic validation passes.
    if (!$form_state->hasAnyErrors()) {
      $this->checkApiConnection($form_state);
    }
  }

  /**
   * Element validation callback for the API URL field.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  public function validateApiUrlElement(array $element, FormStateInterface $form_state): void {
    $apiUrl = trim($element['#value']);

    if (!empty($apiUrl)) {
      // Validate URL format using UrlHelper.
      if (!UrlHelper::isValid($apiUrl, TRUE)) {
        $form_state->setError($element, $this->t('Please enter a valid URL (e.g., https://example.monsido.com/api).'));
        return;
      }

      // Parse URL to check scheme and domain.
      $parsedUrl = parse_url($apiUrl);
      // Enforce HTTPS only.
      if ($parsedUrl['scheme'] !== 'https') {
        $form_state->setError($element, $this->t('API URL must use HTTPS protocol for security.'));
        return;
      }

      // Check against allowed domains.
      $host = strtolower($parsedUrl['host']);
      // Only allow monsido.com sub-domain.
      if ($host !== 'monsido.com' && !str_ends_with($host, '.monsido.com')) {
        $form_state->setError($element, $this->t('API URL must be a valid Acquia Optimize domain (e.g., https://example.monsido.com/api).'));
        return;
      }

      // Remove trailing slash for consistency.
      $apiUrl = rtrim($apiUrl, '/');
      $form_state->setValueForElement($element, $apiUrl);
    }
  }

  /**
   * Tests the API connection with the provided credentials.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state object.
   */
  private function checkApiConnection(FormStateInterface $form_state): void {
    $apiKey = $form_state->getValue('api_key');
    $apiUrl = $form_state->getValue('api_url');

    // Validate the connection using the API client.
    $apiClient = $this->apiClientFactory->create($apiKey, $apiUrl);
    $response = $apiClient->validateApiConnection();

    // Handle API connection errors that should prevent form submission.
    if (isset($response['error'])) {
      $form_state->setErrorByName('', $this->t('API connection failed, @error', ['@error' => $response['error']]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Only proceed with submission if there are no errors.
    if ($form_state->hasAnyErrors()) {
      return;
    }

    $this->config('acquia_optimize.settings')
      ->set('api_key', trim($form_state->getValue('api_key')))
      ->set('api_url', $form_state->getValue('api_url'))
      ->set('accessibility', $form_state->getValue('accessibility'))
      ->set('debug_mode', (bool) $form_state->getValue('debug_mode'))
      ->save();

    // Call parent but suppress its message by clearing it afterwards.
    parent::submitForm($form, $form_state);

    // Replace default message with custom one.
    $this->messenger()->deleteByType('status');
    $this->messenger()->addStatus($this->t('Acquia Optimize settings saved and API connection established successfully.'));
  }

}
