<?php

namespace Drupal\Tests\acquia_optimize\Functional;

use Drupal\acquia_optimize\Enum\ReadabilityData;
use Drupal\acquia_optimize\Enum\SeoIssues;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;

/**
 * Tests the processing and rendering of optimize scan results data.
 *
 * @group acquia_optimize
 */
class OptimizeResultsRenderTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['acquia_optimize'];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Tests the processing of scan results data and template rendering.
   */
  public function testOptimizeResultsDataProcessing() {
    $modulePath = \Drupal::service('extension.path.resolver')->getPath('module', 'acquia_optimize');
    $mockScanResult = json_decode(file_get_contents($modulePath . '/tests/mocks/mockScanResult.json'), TRUE);

    // Test the data processing logic.
    $this->validateScanDataStructure($mockScanResult);
    $this->validateAccessibilityIssues($mockScanResult['accessibility_errors']);
    $this->validateSeoIssues($mockScanResult['seo_issues']);
    $this->validateDataProtectionViolations($mockScanResult['data_protection_violations']);

    // Test template rendering with processed data.
    $build = [
      '#theme' => 'acquia_optimize_modal',
      '#stage' => 'results',
      '#has_error' => FALSE,
      '#config_url' => Url::fromRoute('acquia_optimize.admin_settings')->toString(),
      '#account_url' => 'https://new.monsido.com',
      '#account_text' => 'Go to your Optimize account',
      '#scan_result' => $mockScanResult,
      '#total_issues' => $this->calculateTotalIssues($mockScanResult),
      '#scan_date' => date('F j, Y - g:i A'),
      // Map processed data to template variables.
      '#accessibility_issues' => $mockScanResult['accessibility_errors'] ?? [],
      '#seo_issues' => SeoIssues::getGroupedSeoIssues($mockScanResult['seo_issues'] ?? []),
      '#readability_data' => $this->processReadabilityData($mockScanResult),
      '#data_protection_violations' => $mockScanResult['data_protection_violations'] ?? [],
      '#image_path' => '/' . $modulePath . '/images',
    ];

    $render = \Drupal::service('renderer')->renderInIsolation($build);

    // Test that the template renders without errors.
    $this->assertNotEmpty($render);
    $this->assertStringContainsString('acquia-optimize-modal-content', $render);

    // Test key data elements are present in the rendered output.
    $this->assertStringContainsString('stage-results', $render);
    $this->validateRenderedContent($render, $mockScanResult);
  }

  /**
   * Validates the basic structure of scan result data.
   *
   * @param array $scanResult
   *   The scan result data to validate.
   */
  protected function validateScanDataStructure(array $scanResult): void {
    // Test required top-level keys.
    $requiredKeys = ['accessibility_errors', 'seo_issues', 'data_protection_violations'];
    foreach ($requiredKeys as $key) {
      $this->assertArrayHasKey($key, $scanResult, "Missing required key: $key");
    }

    // Test data types.
    $this->assertIsArray($scanResult['accessibility_errors']);
    $this->assertIsArray($scanResult['seo_issues']);
    $this->assertIsArray($scanResult['data_protection_violations']);
  }

  /**
   * Validates accessibility issues data structure.
   *
   * @param array $accessibilityErrors
   *   The accessibility errors to validate.
   */
  protected function validateAccessibilityIssues(array $accessibilityErrors): void {
    foreach ($accessibilityErrors as $error) {
      // Test required fields for accessibility errors.
      $requiredFields = ['name', 'description', 'level', 'impact'];
      foreach ($requiredFields as $field) {
        $this->assertArrayHasKey($field, $error, "Missing accessibility error field: $field");
      }

      // Test WCAG level values.
      $this->assertContains($error['level'], ['A', 'AA', 'AAA'], 'Invalid WCAG level');

      // Test impact values.
      $this->assertContains($error['impact'], ['minor', 'moderate', 'serious', 'critical'], 'Invalid impact level');
    }
  }

  /**
   * Validates SEO issues data structure.
   *
   * @param array $seoIssues
   *   The SEO issues to validate.
   */
  protected function validateSeoIssues(array $seoIssues): void {
    foreach ($seoIssues as $issue) {
      // Test required fields for SEO issues.
      $this->assertArrayHasKey('name', $issue, 'Missing SEO issue name');
      $this->assertArrayHasKey('classification_name', $issue, 'Missing SEO issue classification');

      // Test classification values.
      $this->assertContains($issue['classification_name'], ['error', 'alert', 'warning', 'info'], 'Invalid SEO classification');
    }
  }

  /**
   * Validates data protection violations data structure.
   *
   * @param array $violations
   *   The data protection violations to validate.
   */
  protected function validateDataProtectionViolations(array $violations): void {
    foreach ($violations as $violation) {
      // Test required fields for data protection violations.
      $requiredFields = ['name', 'content', 'priority'];
      foreach ($requiredFields as $field) {
        $this->assertArrayHasKey($field, $violation, "Missing data protection violation field: $field");
      }

      // Test priority values.
      $this->assertContains($violation['priority'], ['low', 'medium', 'high'], 'Invalid priority level');
    }
  }

  /**
   * Process readability data.
   *
   * @param array $scanResult
   *   The scan result data.
   *
   * @return array
   *   Processed readability data.
   */
  protected function processReadabilityData(array $scanResult): array {
    $readabilityScore = $scanResult['readability'] ?? 0;

    // Use the ReadabilityData enum for processing.
    $readability = ReadabilityData::fromScore($readabilityScore);

    return [
      'score_display' => $readabilityScore,
      'grade_level' => $readability->getGradeLevel()->render(),
      'description' => $readability->getDescription()->render(),
      'score_color' => $readability->getColor(),
    ];
  }

  /**
   * Validates key content appears in the rendered output.
   *
   * @param string $render
   *   The rendered HTML.
   * @param array $scanResult
   *   The original scan result data.
   */
  protected function validateRenderedContent(string $render, array $scanResult): void {
    // Test that issue counts are reflected.
    $totalIssues = $this->calculateTotalIssues($scanResult);
    if ($totalIssues > 0) {
      $this->assertStringContainsString((string) $totalIssues, $render);
    }

    // Test accessibility section if there are errors.
    if (!empty($scanResult['accessibility_errors'])) {
      $this->assertStringContainsString('accessibility', $render);
    }

    // Test SEO section if there are issues.
    if (!empty($scanResult['seo_issues'])) {
      $this->assertStringContainsString('seo', $render);
    }

    // Test data protection section if there are violations.
    if (!empty($scanResult['data_protection_violations'])) {
      $this->assertStringContainsString('data', $render);
    }
  }

  /**
   * Helper method to calculate total issues from scan result.
   *
   * @param array $scanResult
   *   The scan result array.
   *
   * @return int
   *   The total number of issues.
   */
  protected function calculateTotalIssues(array $scanResult): int {
    $total = 0;

    if (!empty($scanResult['accessibility_errors'])) {
      $total += count($scanResult['accessibility_errors']);
    }

    if (!empty($scanResult['seo_issues'])) {
      $total += count($scanResult['seo_issues']);
    }

    if (!empty($scanResult['data_protection_violations'])) {
      $total += count($scanResult['data_protection_violations']);
    }

    return $total;
  }

}
