var defaultErrorHandler = {
  onError: (msg, error) => {
    console.error(msg, error);
  }
};
var PageBuilder = class {
  constructor(errorHandler = defaultErrorHandler) {
    this.errorHandler = errorHandler;
  }
  css = [];
  makePage(content) {
    this.css = content.css;
    const rootCssList = {};
    const rootNode = this.traverseTree(content.dom_tree, rootCssList);
    const docFragment = document.createDocumentFragment();
    if (!rootNode) {
      this.errorHandler.onError("Unable to make page from tree root");
      return docFragment;
    }
    docFragment.appendChild(rootNode);
    const styles = this.buildStyle(rootCssList);
    this.appendStyle(docFragment, styles);
    return docFragment;
  }
  appendStyle(docFragment, styles) {
    const headEl = docFragment.querySelector("head");
    if (headEl) {
      headEl.appendChild(styles);
    } else {
      docFragment.appendChild(styles);
    }
  }
  buildStyle(css) {
    const styleElement = document.createElement("style");
    let styles = "";
    Object.keys(css || {}).forEach((key) => {
      styles += `[data-cs-${key}] {${css[parseInt(key)]}} `;
    });
    styleElement.textContent = styles.trimEnd();
    return styleElement;
  }
  traverseTree(node, cssList) {
    if ("tn" in node) {
      return this.parseElementNode(node, cssList);
    }
    if ("t" in node) {
      return this.parseTextNode(node);
    }
    this.errorHandler.onError("NodeType: Unknown node type");
    return null;
  }
  parseElementNode(node, cssList) {
    let element;
    try {
      element = document.createElement(node.tn || "");
    } catch (error) {
      this.errorHandler.onError(`Invalid Tag name: ${node.tn}`, error);
      return null;
    }
    this.dataCsId(element, cssList, node.ci);
    this.setAttributes(element, node.a);
    this.setShadowDom(element, node.sr);
    this.appendChildren(element, cssList, node.c);
    return element;
  }
  parseTextNode(node) {
    return document.createTextNode(node.t);
  }
  dataCsId(element, cssList, nodeCsId) {
    if (nodeCsId !== void 0) {
      element.setAttribute(`data-cs-0`, "");
      element.setAttribute(`data-cs-${nodeCsId.toString()}`, "");
      cssList[nodeCsId] = this.css[nodeCsId];
    } else {
      this.errorHandler.onError(`Invalid data-cs-id: "${nodeCsId}"`);
    }
  }
  setAttributes(element, attributes) {
    (attributes || []).forEach((attribute) => {
      try {
        element.setAttribute(attribute[0], attribute[1] || "");
      } catch (error) {
        this.errorHandler.onError(`Invalid attribute name: ${attribute[0]}`, error);
      }
    });
  }
  setShadowDom(element, shadowRootNode) {
    if (!shadowRootNode) {
      return;
    }
    let shadowRoot;
    try {
      shadowRoot = element.attachShadow({ mode: "open" });
    } catch (error) {
      this.errorHandler.onError("Shadowroot", error);
      return;
    }
    const shadowRootCssList = {};
    this.appendChildren(shadowRoot, shadowRootCssList, shadowRootNode.c);
    const styles = this.buildStyle(shadowRootCssList);
    shadowRoot.appendChild(styles);
  }
  appendChildren(root, cssList, children) {
    (children || []).forEach((child) => {
      const childNode = this.traverseTree(child, cssList);
      if (childNode) {
        root.appendChild(childNode);
      }
    });
  }
};

var version = "4.1.2";

var DataCollector = class {
  tree = {};
  css = [];
  version;
  disallowedTagNames = ["STYLE", "SCRIPT", "MONSIDO-EXTENSION"];
  monsidoIframeId = "monsido-extension-iframe";
  defaultStyles;
  async collectData(html) {
    const { width, height } = this.getViewPortSize(html);
    if (!width || !height) {
      throw new Error("No viewport size found");
    }
    this.setDefaultComputedStyles();
    const newHtml = this.removeExtensionElements(html);
    const cleanedHtml = this.cleanUpText(newHtml.outerHTML);
    this.tree = await this.processTree(html);
    return { dom_tree: this.tree, css: this.css, html: cleanedHtml, version: this.version || version, viewport: { w: width, h: height } };
  }
  setVersion(v) {
    this.version = v;
  }
  removeExtensionElements(html) {
    const htmlClone = html.cloneNode(true);
    const extensionElements = [`IFRAME#${this.monsidoIframeId}`, this.disallowedTagNames[2]];
    extensionElements.forEach((selector) => {
      const elements = htmlClone.querySelectorAll(selector);
      if (elements) {
        elements.forEach((element) => {
          element.remove();
        });
      }
    });
    return htmlClone;
  }
  processTree(el) {
    return new Promise(async (resolve) => {
      setTimeout(async () => {
        const data = {};
        if (el.nodeType !== 11) {
          data.tn = el.tagName.toUpperCase(), data.ci = this.processStyles(el);
          data.a = this.getAttributesList(el);
          const shadowRoot = el.shadowRoot;
          if (shadowRoot) {
            data.sr = await this.processTree(shadowRoot);
          }
        }
        data.c = [];
        const nodes = Array.from(el.childNodes);
        const lastIndex = nodes.length - 1;
        if (!nodes.length && !data.c.length) {
          delete data.c;
          resolve(data);
        } else {
          for (let i = 0; i < nodes.length; i += 1) {
            const node = nodes[i];
            if (node.nodeType === 1) {
              const tagName = node.tagName.toUpperCase();
              if (this.disallowedTagNames.includes(tagName) || tagName === "IFRAME" && node.getAttribute("id") === this.monsidoIframeId) {
              } else {
                const child = await this.processTree(node);
                data.c.push(child);
              }
            } else if (node.nodeType === 3) {
              data.c.push({
                t: this.cleanUpText(node.textContent || "")
              });
            }
            if (lastIndex === i) {
              if (data.c && !data.c.length) {
                delete data.c;
              }
              resolve(data);
            }
          }
        }
      }, 0);
    });
  }
  setDefaultComputedStyles() {
    this.defaultStyles = this.getStylesAsRecord(document.documentElement);
    this.css.push(this.collectStyles(this.defaultStyles));
  }
  getViewPortSize(html) {
    const viewportWidth = window.visualViewport?.width || window.innerWidth || html.clientWidth;
    const viewportHeight = window.visualViewport?.height || window.innerHeight || html.clientHeight;
    return {
      width: viewportWidth,
      height: viewportHeight
    };
  }
  processStyles(el) {
    const { styles, sameId } = this.collectUniqueStyles(el);
    let csId = 0;
    if (styles) {
      if (sameId === void 0) {
        csId = this.css.length;
        this.css.push(styles);
      } else {
        csId = sameId;
      }
    }
    return csId;
  }
  getStylesAsRecord(el) {
    const styleObj = window.getComputedStyle(el);
    const result = {};
    for (let i = styleObj.length; i--; ) {
      const nameString = styleObj[i];
      result[nameString] = `${styleObj.getPropertyValue(nameString)};`;
    }
    return result;
  }
  collectUniqueStyles(el) {
    const styles = this.collectStyles(this.getStylesAsRecord(el), this.defaultStyles);
    let sameId;
    if (styles.length) {
      const index = this.css.indexOf(styles);
      if (index !== -1) {
        sameId = index;
      }
    }
    return { styles, sameId };
  }
  collectStyles(stylesObj, defaultStyles) {
    if (defaultStyles) {
      stylesObj = this.removeDefaultStyles(stylesObj, defaultStyles);
    }
    return Object.entries(stylesObj).map(([key, value]) => `${this.escapeQuotation(key)}:${this.escapeQuotation(value)}`).join("");
  }
  removeDefaultStyles(stylesObj, defaultObj) {
    const result = {};
    for (const k of Object.keys(stylesObj)) {
      if (stylesObj[k] !== defaultObj[k]) {
        result[k] = stylesObj[k];
      }
    }
    return result;
  }
  getAttributesList(el) {
    const attrNames = Array.from(el.attributes);
    const result = attrNames.map((n) => [n.nodeName, n.nodeValue || ""]);
    return result;
  }
  cleanUpText(text) {
    return text.replaceAll(/ +/g, " ");
  }
  escapeQuotation(text) {
    return text.replaceAll(/"/g, '\\"');
  }
};