<?php

declare(strict_types=1);

namespace Drupal\acquia_optimize\Controller;

use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Controller for Content Quick Scan functionality.
 *
 * Handles the complete workflow for quick scans initiated from content pages.
 */
class ContentQuickScanController extends AcquiaOptimizeController {

  /**
   * The utilities service.
   *
   * @var mixed
   */
  protected $utilities;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->utilities = $container->get('acquia_optimize.utilities');
    $instance->renderer = $container->get('renderer');
    return $instance;
  }

  /**
   * Handles GET requests with integrated polling logic.
   *
   * This method performs server-side polling to check scan status and return
   * results once the scan is completed or failed, eliminating the need for
   * client-side JavaScript polling.
   *
   * @param string $scan_id
   *   The scan ID to check status for.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response with final scan results or error.
   */
  public function getContentQuickScan(string $scan_id): JsonResponse {
    // Polling configuration.
    $polling_config = [
      'max_retries' => 15,
      'max_failure_retries' => 2,
      'polling_interval' => 2,
    ];

    try {
      $final_result = $this->pollScanCompletion($scan_id, $polling_config);
      return new JsonResponse($final_result);
    }
    catch (\Exception $e) {
      $this->logger->error('Server-side polling failed for scan @scan_id: @message', [
        '@scan_id' => $scan_id,
        '@message' => $e->getMessage(),
      ]);

      return new JsonResponse($this->renderErrorResponse(
        $this->t('Server-side polling failed: @message', ['@message' => $e->getMessage()])
          ->render(),
      ));
    }
  }

  /**
   * Performs server-side polling until scan completion.
   *
   * @param string $scan_id
   *   The scan ID to poll for.
   * @param array $config
   *   Polling configuration array.
   * @param int $retry_count
   *   Current retry attempt count.
   * @param int $failure_retry_count
   *   Current failure retry attempt count.
   *
   * @return array
   *   The final scan result array.
   *
   * @throws \Exception
   *   When polling fails or times out.
   */
  private function pollScanCompletion(string $scan_id, array $config, int $retry_count = 0, int $failure_retry_count = 0): array {
    // Check if we've exceeded max polling attempts.
    if ($retry_count >= $config['max_retries']) {
      throw new \Exception(
        $this->t(
        'Polling timeout: Maximum retries (@retries) exceeded for scan @scan_id',
        [
          '@retries' => $config['max_retries'],
          '@scan_id' => $scan_id,
        ])->render()
      );
    }

    try {
      // Get scan status from API.
      $scan_response = $this->getQuickScan($scan_id);
      $scan_result = json_decode($scan_response->getContent(), TRUE);

      return $this->handleScanStatus($scan_result, $scan_id, $config, $retry_count, $failure_retry_count);
    }
    catch (\Exception $e) {
      // ApiClient already handled retries and error mapping.
      throw new \Exception(
        $this->t('API request failed for scan @scan_id after internal retries: @retry_count. Error: @message',
        [
          '@scan_id' => $scan_id,
          '@retry_count' => $retry_count,
          '@message' => $e->getMessage(),
        ])->render()
      );
    }
  }

  /**
   * Handles different scan status values and determines next action.
   *
   * @param array $data
   *   The scan result data.
   * @param string $scan_id
   *   The scan ID.
   * @param array $config
   *   Polling configuration array.
   * @param int $retry_count
   *   Current retry attempt count.
   * @param int $failure_retry_count
   *   Current failure retry attempt count.
   *
   * @return array
   *   The final scan result array or continues polling.
   *
   * @throws \Exception
   *   When polling needs to continue or fails.
   */
  private function handleScanStatus(array $data, string $scan_id, array $config, int $retry_count = 0, int $failure_retry_count = 0): array {
    $status = $data['status'] ?? 'unknown';
    // Handle completed status - success case.
    if ($status === 'completed') {
      // Prepare final result data.
      $final_result = $data;
      if (!empty($data['process_result'])) {
        $final_result = array_merge($data, [
          'id' => $scan_id,
          'total_issues' => $data['process_result']['total_issues'] ?? 0,
          'scan_date' => $data['process_result']['scan_date'] ?? date('D M j, Y'),
          'accessibility_errors' => $data['process_result']['accessibility_errors'] ?? [],
          'seo_issues' => $data['process_result']['seo_issues'] ?? [],
          'readability_data' => $data['process_result']['readability_data'] ?? [],
          'data_protection_violations' => $data['process_result']['data_protection_violations'] ?? [],
        ]);
      }

      return $this->renderModalTemplate($final_result);
    }

    // Handle failed status.
    if ($status === 'failed' || isset($data['error'])) {
      $error_message = (string) ($data['failure_reason'] ?? $data['error_message'] ?? $data['error'] ?? $this->t('Scan failed'));

      // Check if we should retry the failed scan.
      if ($failure_retry_count < $config['max_failure_retries']) {
        // Wait before retrying the failed scan.
        sleep($config['polling_interval']);
        return $this->pollScanCompletion($scan_id, $config, 0, $failure_retry_count + 1);
      }
      else {
        return $this->renderErrorResponse($error_message);
      }
    }

    // Handle other explicit failure statuses without retries.
    if (in_array($status, ['error', 'cancelled', 'timeout'])) {
      $error_message = (string) ($data['error'] ?? $data['message'] ?? $this->t("Scan @status", ['@status' => $status]));
      return $this->renderErrorResponse($error_message);
    }

    // Continue polling for in-progress statuses.
    sleep($config['polling_interval']);
    return $this->pollScanCompletion($scan_id, $config, $retry_count + 1, $failure_retry_count);
  }

  /**
   * Renders error response in the modal template format.
   *
   * @param string $error_message
   *   The error message to display.
   * @param int $status_code
   *   The HTTP status code.
   *
   * @return array
   *   The response array with rendered error template.
   */
  private function renderErrorResponse(string $error_message, int $status_code = 500): array {
    $error_data = [
      'error_message' => $error_message,
      'status_code' => $status_code,
    ];
    return $this->renderModalTemplate($error_data, TRUE);
  }

  /**
   * Renders the acquia-optimize-modal template with provided variables.
   *
   * @param array $data
   *   Template variables to pass to the modal template.
   * @param bool $is_failed
   *   Whether the scan failed, used to adjust template rendering.
   *
   * @return array
   *   Variable array for the modal template.
   */
  private function renderModalTemplate(array $data, bool $is_failed = FALSE): array {
    $build = [
      '#theme' => 'acquia_optimize_modal',
      '#image_path' => $this->utilities->getImagePath(),
      '#config_url' => Url::fromRoute('acquia_optimize.admin_settings')->toString(),
      '#account_url' => 'https://new.monsido.com',
      '#account_text' => $this->t('Go to your Web Governance account'),
      '#attached' => [
        'library' => ['acquia_optimize/optimize'],
      ],
    ];
    if ($is_failed) {
      $output = [
        'status' => 'failed',
        'error_message' => $data['error_message'],
        'status_code' => $data['status_code'] ?? 500,
      ];
      $build = array_merge($build, [
        '#stage' => 'results',
        '#has_error' => TRUE,
        '#error_message' => $data['error_message'] ?? (string) $this->t('An unexpected error occurred Please try again later.'),
      ]);
    }
    else {
      $output = [
        'scan_result' => $data['scan_result'] ?? [],
        'status' => $data['status'] ?? 'completed',
      ];
      $build = array_merge($build, [
        '#stage' => 'results',
        '#has_error' => FALSE,
        '#scan_id' => $data['id'] ?? NULL,
        '#total_issues' => $data['total_issues'] ?? 0,
        '#scan_date' => $data['scan_date'] ?? date('D M j, Y'),
        '#accessibility_issues' => $data['accessibility_errors'] ?? [],
        '#seo_issues' => $data['seo_issues'] ?? [],
        '#readability_data' => $data['readability_data'] ?? [],
        '#data_protection_violations' => $data['data_protection_violations'] ?? [],
      ]);
    }
    $variable_data = $this->renderer->renderInIsolation($build);

    return array_merge($output, [
      'rendered_results' => $variable_data,
    ]);
  }

}
