<?php

declare(strict_types=1);

namespace Drupal\acquia_optimize\Enum;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Readability difficulty levels based on LIX Ease scores.
 */
enum ReadabilityData: string {
  case VeryEasy = 'very_easy';
  case Easy = 'easy';
  case Medium = 'medium';
  case Difficult = 'difficult';
  case VeryDifficult = 'very_difficult';

  /**
   * Get the grade level description.
   */
  public function getGradeLevel(): TranslatableMarkup {
    return match($this) {
      self::VeryEasy => new TranslatableMarkup('Very Easy'),
      self::Easy => new TranslatableMarkup('Easy'),
      self::Medium => new TranslatableMarkup('Medium'),
      self::Difficult => new TranslatableMarkup('Difficult'),
      self::VeryDifficult => new TranslatableMarkup('Very difficult'),
    };
  }

  /**
   * Get the detailed description for this readability level.
   */
  public function getDescription(): TranslatableMarkup {
    return match($this) {
      self::VeryEasy => new TranslatableMarkup('Text is extremely simple and straightforward, using short sentences and very common words. Suitable for young children or beginners in the language.'),
      self::Easy => new TranslatableMarkup('Text is simple, with mostly short sentences and familiar vocabulary. Most readers will have little to no difficulty understanding the content.'),
      self::Medium => new TranslatableMarkup('Text has moderately complex sentence structures and some less common vocabulary. Suitable for general audiences with average reading skills.'),
      self::Difficult => new TranslatableMarkup('Text is quite challenging, with long sentences and more advanced vocabulary. Suitable for readers with strong reading skills or subject matter knowledge.'),
      self::VeryDifficult => new TranslatableMarkup('Text is highly complex, featuring very long sentences, infrequent or technical vocabulary, and advanced structure. Best suited for experts or specialized audiences.'),
    };
  }

  /**
   * Get the color for UI display based on readability quality.
   */
  public function getColor(): string {
    return match($this) {
      self::VeryEasy, self::Easy => 'rgb(92, 113, 57)',
      self::Medium => 'rgb(255, 165, 0)',
      self::Difficult, self::VeryDifficult => 'rgb(215, 58, 73)',
    };
  }

  /**
   * Create ReadabilityLevel from score.
   */
  public static function fromScore(int|float $score): self {
    return match (TRUE) {
      $score >= 55 => self::VeryDifficult,
      $score >= 45 => self::Difficult,
      $score >= 35 => self::Medium,
      $score >= 25 => self::Easy,
      default => self::VeryEasy,
    };
  }

}
