<?php

declare(strict_types=1);

namespace Drupal\acquia_optimize\Enum;

use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Enum for SEO issues with their friendly names.
 */
enum SeoIssues: string {

  case MissingTitle = 'seo_missing_title';
  case DuplicateTitle = 'seo_duplicate_title';
  case TooLongUrl = 'seo_too_long_url';
  case MissingH1 = 'seo_missing_h1';
  case MultipleH1 = 'seo_multiple_h1';
  case DuplicateH1 = 'seo_duplicate_h1';
  case MissingMetaDescription = 'seo_missing_meta_description';
  case TooShortMetaDescription = 'seo_too_short_meta_description';
  case TooLongMetaDescription = 'seo_too_long_meta_description';
  case TooManyPageLinks = 'seo_too_many_page_links';
  case MissingSubheadlines = 'seo_missing_subheadlines';
  case MissingAlt = 'seo_missing_alt';
  case NoIndex = 'seo_no_index';
  case NoFollow = 'seo_no_follow';

  /**
   * Get the friendly name for the SEO issue.
   *
   * @return string
   *   The friendly name of the SEO issue.
   */
  public function getFriendlyName(): TranslatableMarkup {
    return match ($this) {
      self::MissingTitle => new TranslatableMarkup('Missing title'),
      self::DuplicateTitle => new TranslatableMarkup('Title found on more than one page'),
      self::TooLongUrl => new TranslatableMarkup('Too long URL'),
      self::MissingH1 => new TranslatableMarkup('Missing H1'),
      self::MultipleH1 => new TranslatableMarkup('Multiple H1 on page'),
      self::DuplicateH1 => new TranslatableMarkup('H1 found on more than one page'),
      self::MissingMetaDescription => new TranslatableMarkup('Missing META description'),
      self::TooShortMetaDescription => new TranslatableMarkup('Too short META description'),
      self::TooLongMetaDescription => new TranslatableMarkup('Too long META description'),
      self::TooManyPageLinks => new TranslatableMarkup('Too many internal links'),
      self::MissingSubheadlines => new TranslatableMarkup('Missing sub headings'),
      self::MissingAlt => new TranslatableMarkup('Images missing ALT'),
      self::NoIndex => new TranslatableMarkup('Pages with - No index'),
      self::NoFollow => new TranslatableMarkup('Links with - No follow'),
    };
  }

  /**
   * Group SEO issues into processed format with friendly names and categories.
   *
   * @param array $seoIssues
   *   The raw SEO issues array.
   *
   * @return array
   *   Array of processed SEO issues with friendly names and categories.
   */
  public static function getGroupedSeoIssues(array $seoIssues): array {
    $processed_issues = [];

    foreach ($seoIssues as $issue) {
      $seoIssueEnum = self::tryFrom($issue['name']);
      $issue['friendly_name'] = $seoIssueEnum?->getFriendlyName()->render() ?? $issue['name'];
      $issue['category'] = match($issue['classification_name']) {
        'error', 'alert' => 'high',
        'warning' => 'medium',
        'info' => 'low',
        'technical' => 'technical',
        default => 'medium',
      };
      $processed_issues[] = $issue;
    }

    return $processed_issues;
  }

}
