<?php

declare(strict_types=1);

namespace Drupal\acquia_optimize\Form;

use Drupal\acquia_optimize\Utilities;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenDialogCommand;
use Drupal\Core\Ajax\PrependCommand;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Ajax\RemoveCommand;
use Drupal\Core\Ajax\ScrollTopCommand;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\TempStore\TempStoreException;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides form alteration and AJAX callback logic for Quick scan.
 *
 * This class is intended to be used as a service to alter node forms and handle
 * AJAX interactions for the quick scan.
 */
class AcquiaOptimizeFormAlter implements ContainerInjectionInterface {
  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * Constructs a AcquiaOptimizeFormAlter object.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $tempStoreFactory
   *   The private temp store factory.
   * @param \Drupal\acquia_optimize\Utilities $utilities
   *   The Acquia Web Governance utilities service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $currentRouteMatch
   *   The currently active route match object.
   */
  public function __construct(
    private readonly AccountInterface $currentUser,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly MessengerInterface $messenger,
    private readonly RendererInterface $renderer,
    private readonly PrivateTempStoreFactory $tempStoreFactory,
    private readonly Utilities $utilities,
    private readonly RouteMatchInterface $currentRouteMatch,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('current_user'),
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('renderer'),
      $container->get('tempstore.private'),
      $container->get('acquia_optimize.utilities'),
      $container->get('current_route_match'),
    );
  }

  /**
   * Performs the needed alterations to the form.
   *
   * @param array $form
   *   The form to be altered to provide the acquia_optimize form support.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function formAlter(array &$form, FormStateInterface $form_state): void {
    if (!$this->currentUser->hasPermission('scan acquia optimize') ||
    str_starts_with($this->currentRouteMatch->getRouteName(), 'experience_builder.api.')
    ) {
      return;
    }
    // Add required libraries first.
    $form['#attached']['library'][] = 'acquia_optimize/optimize';

    $form['acquia_optimize'] = [
      '#type' => 'container',
      '#group' => 'advanced',
      '#attributes' => [
        'class' => ['acquia-optimize-simple-container'],
        'id' => 'acquia-optimize-wrapper',
      ],
    ];

    $button_text = $this->t("Configure API");
    $ajax_callback = 'acquiaOptimizeAjaxConfigureCallback';
    if ($this->utilities->apiKeyHasBeenSet()) {
      $button_text = $this->t("Quick Scan");
      $ajax_callback = 'acquiaOptimizeAjaxCallback';
    }

    // Single button that changes behavior based on API key status.
    $form['acquia_optimize']['scan'] = [
      '#type' => 'button',
      '#value' => $button_text,
      '#ajax' => [
        'callback' => [$this, $ajax_callback],
        'event' => 'click',
        'wrapper' => 'acquia-optimize-wrapper',
        'progress' => [
          'type' => 'none',
        ],
      ],
      '#attributes' => [
        'class' => ['acquia-optimize-scan-button'],
      ],
    ];
  }

  /**
   * Handles the AJAX callback for the Acquia Web Governance scan button.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function acquiaOptimizeAjaxCallback(array &$form, FormStateInterface $form_state): AjaxResponse {
    $this->triggerFormValidation($form, $form_state);

    if ($form_state->hasAnyErrors()) {
      return $this->getErrorResponse();
    }

    $form_object = $form_state->getFormObject();

    // Build entity without triggering submit handlers.
    $node = $form_object->buildEntity($form, $form_state);

    // Set the entity back to the form object.
    $form_object->setEntity($node);

    try {
      $this->saveFormState($node, $form_state);
    }
    catch (TempStoreException $e) {
      $this->messenger->addError($this->t('Unable to save form state for preview. @error_message', [
        '@error_message' => $e->getMessage(),
      ]));
      // Optionally log the exception.
      return $this->getErrorResponse();
    }

    return $this->openScanDialog($this->buildScanForm($node));
  }

  /**
   * Handles the AJAX callback to redirect to configuration page.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function acquiaOptimizeAjaxConfigureCallback(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();

    // Get the clean current path without AJAX parameters.
    $route_name = $this->currentRouteMatch->getRouteName();
    $route_parameters = $this->currentRouteMatch->getRawParameters()->all();

    // Build clean destination URL.
    $current_url = Url::fromRoute($route_name, $route_parameters)->toString();

    // Build the redirect URL to the configuration page with destination.
    $config_url = Url::fromRoute('acquia_optimize.admin_settings', [], [
      'query' => ['destination' => $current_url],
    ])->toString();

    $response->addCommand(new RedirectCommand($config_url));

    return $response;
  }

  /**
   * Builds the error messages array for rendering.
   */
  private function getErrorMessages(): array {
    $messages = $this->messenger->all();
    $this->messenger->deleteAll();

    return [
      '#theme' => 'status_messages',
      '#message_list' => $messages,
    ];
  }

  /**
   * Triggers validation for the node form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  private function triggerFormValidation(array &$form, FormStateInterface $form_state): void {
    /** @var \Drupal\node\NodeForm $form_object */
    $form_object = $form_state->getBuildInfo()['callback_object'];
    if (!$form_state->isSubmitted()) {
      $form_state->setSubmitted();
    }
    $form_state->setValidationComplete(FALSE);
    $form_object->validateForm($form, $form_state);
  }

  /**
   * Builds an AJAX response containing error messages.
   */
  private function getErrorResponse(): AjaxResponse {
    $response = new AjaxResponse();
    $error_output = $this->getErrorMessages();

    // Remove existing messages, prepend new ones, and scroll to top.
    $response->addCommand(new RemoveCommand('.messages-list'));
    $response->addCommand(new PrependCommand(
      '.node-form',
      $this->renderer->render($error_output)
    ));
    $response->addCommand(new ScrollTopCommand('body'));

    return $response;
  }

  /**
   * Saves the form state in the private temp store for node preview.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @throws \Drupal\Core\TempStore\TempStoreException
   */
  private function saveFormState(NodeInterface $node, FormStateInterface $form_state): void {
    $temp_store = $this->tempStoreFactory->get('node_preview');
    $node->in_preview = TRUE;
    $temp_store->set($node->uuid(), $form_state);
  }

  /**
   * Builds the scan form modal output.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node entity.
   */
  private function buildScanForm(NodeInterface $node): array {
    $preview_url = Url::fromRoute('acquia_optimize.node_preview', [
      'node_preview' => $node->uuid(),
      'view_mode_id' => 'full',
    ]);
    $config = $this->configFactory->get('acquia_optimize.settings');
    return [
      'scan_form' => [
        '#theme' => 'acquia_optimize_modal',
        '#image_path' => $this->utilities->getImagePath(),
        '#config_url' => Url::fromRoute('acquia_optimize.admin_settings')->toString(),
        '#account_url' => 'https://new.monsido.com',
        '#account_text' => $this->t('Go to your Web Governance account'),
        '#attached' => [
          'library' => ['acquia_optimize/optimize'],
          'drupalSettings' => [
            'acquiaOptimize' => [
              'previewUrl' => $preview_url->toString(),
              'nodeUuid' => $node->uuid(),
              'accessibility' => $config->get('accessibility') ?: 'WCAG21-AA',
            ],
          ],
        ],
      ],
    ];
  }

  /**
   * Opens the scan dialog with the provided output.
   *
   * @param array $output
   *   The render array for the dialog content.
   */
  private function openScanDialog(array $output): AjaxResponse {
    $response = new AjaxResponse();
    $response->addCommand(new OpenDialogCommand(
      '#acquia-optimize-scan-container',
      $this->t('Acquia Web Governance'),
      $output,
      [
        'width' => 500,
        'height' => 'auto',
        'modal' => TRUE,
        'dialogClass' => 'acquia-optimize-dialog',
        'resizable' => FALSE,
        'draggable' => TRUE,
      ],
    ));
    return $response;
  }

}
