<?php

namespace Drupal\acquia_optimize;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Session\AccountProxyInterface;

/**
 * Contains utility functions for module.
 */
class Utilities {

  /**
   * Constructor for the Utilities class.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly AccountProxyInterface $currentUser,
  ) {}

  /**
   * Accessibility guidelines.
   *
   * @var array
   */
  public const ACCESSIBILITY_GUIDELINES = [
    'WCAG2-A',
    'WCAG2-AA',
    'WCAG2-AAA',
    'WCAG21-A',
    'WCAG21-AA',
    'WCAG21-AAA',
    'WCAG22-A',
    'WCAG22-AA',
    'WCAG22-AAA',
  ];

  /**
   * Get image path.
   */
  public function getImagePath(?string $imageName = NULL): string {
    $modulePath = $this->moduleHandler->getModule('acquia_optimize')->getPath();
    // Exact module path without suffix.
    $cleanModulePath = preg_replace('/-\d+$/', '', $modulePath);
    $base_directory_path = DIRECTORY_SEPARATOR . $cleanModulePath . DIRECTORY_SEPARATOR . 'images';
    $path = $imageName ? $base_directory_path . DIRECTORY_SEPARATOR . $imageName : $base_directory_path;
    return is_link($path) ? realpath($path) : $path;
  }

  /**
   * Check if an API key has been set.
   */
  public function apiKeyHasBeenSet(): bool {
    return !empty($this->configFactory->get('acquia_optimize.settings')->get('api_key'));
  }

  /**
   * Check if user can configure API settings.
   */
  public function canConfigureApiSettings(): bool {
    return $this->currentUser->hasPermission('administer acquia optimize');
  }

}
