import { debugError } from '../utils/constants';

export class PollingService {
  constructor(apiService, isDebugMode = false) {
    this.apiService = apiService;
    this.isDebugMode = isDebugMode;
    this.maxRetries = 10; // Maximum number of polling attempts
    this.maxFailureRetries = 2; // Maximum number of retries for failed scans
    this.pollingInterval = 2000; // 2 seconds between polls
  }

  /**
   * Polls scan results until completion or failure
   * @param {string} scanId - The scan ID to poll
   * @param {number} progressInterval - Progress animation interval ID
   * @param {Function} setScanProgress - Progress update callback
   * @param {Function} setScanResults - Results update callback
   * @param {Function} setCurrentView - View update callback
   * @param {number} retryCount - Current retry attempt (internal use)
   * @param {number} failureRetryCount - Current failure retry attempt (internal use)
   */
  async pollScanResults(scanId, progressInterval, setScanProgress, setScanResults, setCurrentView, retryCount = 0, failureRetryCount = 0) {
    try {
      if (!scanId) {
        throw new Error('Invalid scan ID');
      }

      // Check if we've exceeded max retries
      if (retryCount >= this.maxRetries) {
        debugError(this.isDebugMode, `Polling timeout: Maximum retries (${this.maxRetries}) exceeded after ${this.maxRetries * this.pollingInterval / 1000} seconds`);
      }

      // Check scan status - apiService.checkScanStatus returns the data directly, not a Response object
      const data = await this.apiService.checkScanStatus(scanId, this.isDebugMode);

      if (data) {
        const status = data.status;

        this.handleScanStatus(
          status,
          data,
          scanId,
          progressInterval,
          setScanProgress,
          setScanResults,
          setCurrentView,
          retryCount,
          failureRetryCount
        );
      } else {
        // Handle case where checkScanStatus returns null - retry instead of failing immediately
        this.retryPolling(
          scanId,
          progressInterval,
          setScanProgress,
          setScanResults,
          setCurrentView,
          retryCount,
          failureRetryCount,
          'No data received from scan status check'
        );
      }
    } catch (error) {
      // Check if this is a network/temporary error that should be retried
      if (this.shouldRetryError(error) && retryCount < this.maxRetries) {
        this.retryPolling(
          scanId,
          progressInterval,
          setScanProgress,
          setScanResults,
          setCurrentView,
          retryCount,
          failureRetryCount,
          error.message
        );
      } else {
        this.handlePollingException(
          error,
          progressInterval,
          setScanProgress,
          setScanResults,
          setCurrentView
        );
      }
    }
  }

  /**
   * Handles different scan status values - continues polling until 'completed' status
   * @param {string} status - Current scan status
   * @param {Object} data - Full response data
   * @param {string} scanId - Scan ID
   * @param {number} progressInterval - Progress animation interval
   * @param {Function} setScanProgress - Progress update callback
   * @param {Function} setScanResults - Results update callback
   * @param {Function} setCurrentView - View update callback
   * @param {number} retryCount - Current retry count
   * @param {number} failureRetryCount - Current failure retry count
   */
  handleScanStatus(status, data, scanId, progressInterval, setScanProgress, setScanResults, setCurrentView, retryCount = 0, failureRetryCount = 0) {
    // Only 'completed' status stops polling - all others continue
    const completedStatuses = ['completed'];
    const explicitFailureStatuses = ['failed', 'error', 'cancelled', 'timeout'];

    // Continue polling for scanning, processing, and any other status except completed/failed
    const shouldContinuePolling = !completedStatuses.includes(status) && !explicitFailureStatuses.includes(status) && !data.error;

    if (shouldContinuePolling) {
      // Continue polling for any non-completed status
      setTimeout(() => {
        this.pollScanResults(scanId, progressInterval, setScanProgress, setScanResults, setCurrentView, retryCount + 1, failureRetryCount);
      }, this.pollingInterval);

    } else if (completedStatuses.includes(status)) {
      // Only stop when we get 'completed' status
      this.handleScanCompletion(data, progressInterval, setScanProgress, setScanResults, setCurrentView);

    } else {
      // Handle explicit failures with retry logic
      if (status === 'failed' && failureRetryCount < this.maxFailureRetries) {
        if (this.isDebugMode) {
          debugError(this.isDebugMode, `Scan failed (attempt ${failureRetryCount + 1}/${this.maxFailureRetries}), retrying in ${this.pollingInterval}ms...`);
        }

        // Retry the failed scan
        setTimeout(() => {
          this.pollScanResults(scanId, progressInterval, setScanProgress, setScanResults, setCurrentView, 0, failureRetryCount + 1);
        }, this.pollingInterval);
      } else {
        // Final failure after all retries or non-retryable failure
        if (this.isDebugMode) {
          if (status === 'failed' && failureRetryCount >= this.maxFailureRetries) {
            debugError(this.isDebugMode, `Scan failed permanently after ${failureRetryCount} retries`);
          } else {
            debugError(this.isDebugMode, `Scan failed with status: ${status}`);
          }
        }
        this.handleScanFailure(data, progressInterval, setScanProgress, setScanResults, setCurrentView);
      }
    }
  }

  /**
   * Handles successful scan completion
   */
  handleScanCompletion(data, progressInterval, setScanProgress, setScanResults, setCurrentView) {
    clearInterval(progressInterval);
    setScanProgress(100);
    // Use process_result if available (contains processed data from PHP), 
    // otherwise fall back to scan_result or raw data
    const resultData = data.process_result || data.scan_result || data;
    setScanResults(resultData);
    setTimeout(() => {
      setCurrentView('results');
    }, 500);
  }

  /**
   * Handles scan failure
   */
  handleScanFailure(data, progressInterval, setScanProgress, setScanResults, setCurrentView) {
    clearInterval(progressInterval);
    setScanProgress(100);
    setScanResults({
      status: 'failed',
      error: true,
      message: data.failure_reason || data.message || data.error_message || 'Scan failed',
    });
    setTimeout(() => {
      setCurrentView('results');
    }, 500);
  }

  /**
   * Handles unknown scan status - now removed since we continue polling for any non-completed status
   */
  handleUnknownStatus(status, data, scanId, progressInterval, setScanProgress, setScanResults, setCurrentView, retryCount = 0, failureRetryCount = 0) {
    // Continue polling for unknown statuses unless we've exceeded retry limit
    if (retryCount < this.maxRetries) {
      if (this.isDebugMode) {
        debugError(this.isDebugMode, `Unknown status "${status}", continuing to poll (attempt ${retryCount + 1}/${this.maxRetries})...`);
      }

      setTimeout(() => {
        this.pollScanResults(scanId, progressInterval, setScanProgress, setScanResults, setCurrentView, retryCount + 1, failureRetryCount);
      }, this.pollingInterval);
    } else {
      // Stop polling and show results with warning after max retries
      if (this.isDebugMode) {
        debugError(this.isDebugMode, `Unknown status "${status}", stopping polling after ${retryCount} attempts`);
      }

      clearInterval(progressInterval);
      setScanProgress(100);

      const results = data.scan_result ?
        data.scan_result :
        { ...data, warning: `Scan timed out with status: ${status}` };

      setScanResults(results);
      setTimeout(() => {
        setCurrentView('results');
      }, 500);
    }
  }

  /**
   * Retry polling with exponential backoff
   */
  retryPolling(scanId, progressInterval, setScanProgress, setScanResults, setCurrentView, retryCount, failureRetryCount, reason) {
    const backoffDelay = Math.min(this.pollingInterval * Math.pow(1.2, Math.min(retryCount, 5)), 5000); // Max 5 seconds

    setTimeout(() => {
      this.pollScanResults(scanId, progressInterval, setScanProgress, setScanResults, setCurrentView, retryCount + 1, failureRetryCount);
    }, backoffDelay);
  }

  /**
   * Determine if an error should trigger a retry
   */
  shouldRetryError(error) {
    // Retry on network errors, timeouts, and server errors
    const retryableErrors = [
      'NetworkError',
      'TimeoutError',
      'fetch',
      '5', // 5xx server errors
      'timeout',
      'network',
      'connection',
      'ECONNRESET',
      'ENOTFOUND'
    ];

    const errorString = (error.message || error.toString()).toLowerCase();
    return retryableErrors.some(keyword => errorString.includes(keyword));
  }

  /**
   * Handles polling exceptions
   */
  handlePollingException(error, progressInterval, setScanProgress, setScanResults, setCurrentView) {
    debugError(this.isDebugMode, 'Error polling scan results:', error);
    clearInterval(progressInterval);
    setScanProgress(100);

    // Handle ApiError instances with more specific error information
    let errorMessage = 'Failed to check scan status';
    let errorDetails = {};

    if (error.name === 'ApiError') {
      errorMessage = error.message;
      errorDetails = error.getErrorDetails ? error.getErrorDetails() : {};
    } else {
      errorMessage = error.message || errorMessage;
    }

    setScanResults({
      status: 'failed',
      error: true,
      message: errorMessage,
      ...errorDetails
    });
    setTimeout(() => {
      setCurrentView('results');
    }, 500);
  }

}
