import { HtmlCssCollector } from './HtmlCssCollector';
import { debugError } from '../utils/constants';

export class ScanService {
  constructor(apiService, isDebugMode = false) {
    this.apiService = apiService;
    this.isDebugMode = isDebugMode;
    this.htmlCssCollector = new HtmlCssCollector(isDebugMode);
  }

  /**
   * Starts a quick scan and returns scan information
   * @param {Object} layout - Redux layout state
   * @param {Function} setScanProgress - Progress update function
   * @param {boolean} isRetry - Whether this is a retry attempt
   * @returns {Promise<{scanId: string, progressInterval: number}>}
   */
  async startQuickScan(layout, setScanProgress, isRetry = false, accessibilityGuideline = 'WCAG21-AA') {
    try {
      // Collect HTML and CSS content with the new structured format
      const pageData = await this.htmlCssCollector.collectPageContent(layout);

      // Start progress animation
      const progressInterval = this.startProgressAnimation(setScanProgress);

      // Prepare API request data using the collected page data
      const requestBody = await this.prepareRequestBody(pageData, accessibilityGuideline);

      // Make API call to start scan
      const response = await this.apiService.startQuickScan(requestBody);

      if (response.ok) {
        const data = await response.json();
        const scanId = data.id || '';
        return { scanId, progressInterval };
      } else {
        // Handle API error
        const errorData = await this.handleApiError(response);
        throw new Error(errorData.message);
      }
    } catch (error) {
      debugError(this.isDebugMode, 'Error starting quick scan:', error);
      throw error;
    }
  }

  /**
   * Starts the progress animation
   * @param {Function} setScanProgress - Progress update function
   * @returns {number} - Interval ID
   */
  startProgressAnimation(setScanProgress) {
    let progress = 0;

    return setInterval(() => {
      // Initial fast progress up to 60%
      if (progress < 60) {
        progress += 2;
      }
      // Slow down between 60-80%
      else if (progress < 80) {
        progress += 0.5;
      }
      // Very slow progress after 80%
      else if (progress < 90) {
        progress += 0.1;
      }
      // Add slight animation at the end
      else if (progress < 94) {
        const fluctuation = (Math.random() * 0.3) - 0.1;
        progress += fluctuation;
        progress = Math.min(Math.max(progress, 90), 94);
      }
      setScanProgress(progress);
    }, 500);
  }

  /**
   * Prepares the request body for the API call
   * @param {Object} pageData - Page data from HtmlCssCollector {dom_tree, css, html, version, viewport}
   * @param {string} accessibilityGuideline - Accessibility guideline to use
   * @returns {Object} - Request body
   */
  async prepareRequestBody(pageData, accessibilityGuideline) {
    // Process HTML data according to Web Governance API requirements
    const encodedPageData = {
      dom_tree: pageData.dom_tree,
      css: pageData.css, // pageData.css is already an array
      version: pageData.version,
      viewport: pageData.viewport
    };

    return {
      encoded_page: encodedPageData,
      html: pageData.html,
      css: pageData.css.join('\n'), // Convert array to string for the html field
      accessibility: accessibilityGuideline
    };
  }

  /**
   * Handles API error responses
   * @param {Response} response - API response
   * @returns {Promise<{message: string}>} - Error data
   */
  async handleApiError(response) {
    let errorData;
    let errorMessage;

    try {
      errorData = await response.json();
      debugError(this.isDebugMode, 'Optimize API Error (JSON):', errorData);

      // Special handling for token error
      if (response.status === 401 ||
          (errorData.message && errorData.message.includes('token')) ||
          (errorData.error && errorData.error.includes('token'))) {
        errorMessage = 'Authentication failed: A valid API token must be provided';
      } else {
        errorMessage = errorData.failure_reason || errorData.message || errorData.error_message ||
          `Failed to start scan: ${response.status} ${response.statusText}`;
      }
    } catch (e) {
      // Not JSON, try to get as text
      const textResponse = await response.text();
      debugError(this.isDebugMode, 'Optimize API Error (Text):', textResponse);

      if (textResponse.includes('token')) {
        errorMessage = 'Authentication failed: A valid API token must be provided';
      } else {
        errorMessage = `Failed to start scan: ${response.status} ${response.statusText}`;
      }

      errorData = { error: textResponse };
    }

    return { ...errorData, message: errorMessage };
  }
}
