# Contributing to Acquia Web Governance

Thank you for your interest in contributing to the Acquia Web Governance module! This guide will help you get started with development and contributions.

## Getting Started

### Prerequisites
- PHP 8.1+ (recommended)
- Node.js 18+ (for UI development)
- npm 10+
- Drupal 10.3+ development environment
- Composer 2.0+
- Development tools: DDEV or similar (DDEV recommended)

### Local Development Setup

1. **Clone the repository**:
```bash
git clone git@git.drupal.org:project/acquia_optimize.git
cd acquia_optimize
```

2. **Install dependencies**:
```bash
composer install
```

3. **Set up your Drupal environment**:
```bash
ddev start
ddev composer require drupal/acquia_optimize
ddev drush en acquia_optimize
```

## Building the UI

The Acquia Web Governance module includes a React-based UI for the Canvas integration. To build the UI:

```bash
cd ui
npm install
npm run build
```

For development with hot reloading:
```bash
npm run dev
```

This will compile the React components into the `ui/dist/bundle.js` file that gets loaded by Drupal.

## Development Workflow

1. **Make your code changes**
2. **Build the UI components** (if UI changes were made)
3. **Clear the Drupal cache**: `ddev drush cr`
4. **Test your changes** in both regular content editing and Canvas interface
5. **Run tests** to ensure no regressions
6. **Check coding standards** compliance

## Architecture Overview

Understanding the module structure will help with contributions:

### Core Components

- **Controllers**:
  - `src/Controller/AcquiaOptimizeController.php` - AJAX endpoints and API communication
  - `src/Controller/AcquiaOptimizePreviewController.php` - Node preview generation
  - `src/Controller/ContentQuickScanController.php` - Content-specific scan handling

- **Forms & UI Integration**:
  - `src/Form/SettingsForm.php` - Configuration interface with API validation
  - `src/Form/AcquiaOptimizeFormAlter.php` - Content editing form integration and modal handling

- **Services & Utilities**:
  - `src/ApiClient.php` and `src/ApiClientFactory.php` - External API integration
  - `src/Utilities.php` - Core processing logic and data transformation

- **Frontend Components**:
  - `templates/acquia-optimize-modal.html.twig` - Progressive modal template with three stages
  - `js/quickScan.js` - Content page modal interactions and AJAX handling
  - `ui/` - Complete React application for Canvas integration

### Modal Architecture

The content page integration uses a three-stage modal system:

1. **Initial Stage**: Scan initiation and API configuration access
2. **Scanning Stage**: Real-time progress indication with spinner
3. **Results Stage**: Comprehensive, collapsible results display

### Data Flow

1. **Content Integration**: Form alter adds scan button to node edit forms
2. **AJAX Processing**: Form state saved to temp store, preview URL generated
3. **Modal Rendering**: Drupal modal opened with initial stage template
4. **Scan Execution**: JavaScript initiates API call via AJAX endpoints
5. **Progressive Updates**: Modal stages update based on scan progress
6. **Result Display**: Categorized results rendered in expandable sections

## Testing

### Running Tests

**PHPUnit tests**:
```bash
ddev exec XDEBUG_MODE=coverage phpunit web/modules/contrib/acquia_optimize/tests
```
### Writing Tests

When adding new features, please include appropriate test coverage:

#### Content Page Modal Tests
- **Form Integration Tests**: Test button states and AJAX callbacks
- **Modal Rendering Tests**: Verify template output for all stages
- **JavaScript Tests**: Test modal interactions and state transitions
- **Workflow Tests**: End-to-end scanning process validation

#### API Integration Tests
- **Endpoint Tests**: Test all AJAX endpoints with various inputs
- **Error Handling Tests**: Verify proper error responses and user feedback
- **Permission Tests**: Ensure proper access control
- **Mock Response Tests**: Use provided mock data for predictable testing

#### User Interface Tests
- **Accessibility Tests**: Verify ARIA compliance and keyboard navigation
- **Responsive Tests**: Test modal display across different screen sizes
- **Cross-browser Tests**: Ensure compatibility with supported browsers
- **Performance Tests**: Verify acceptable loading and response times

## Coding Standards

### PHP Code
- Follow [Drupal coding standards](https://www.drupal.org/docs/develop/standards)
- Use PHP_CodeSniffer: `phpcs --standard=Drupal,DrupalPractice`
- Use PHP Code Beautifier: `phpcbf --standard=Drupal,DrupalPractice`

### JavaScript/React Code
- Follow React best practices
- Use ESLint for code quality
- Use Prettier for code formatting
- Follow modern JavaScript (ES6+) patterns

### CSS/SCSS
- Follow Drupal CSS coding standards
- Use BEM methodology for class naming
- Ensure responsive design principles

## API Integration

When working with API features:

1. **Use the ApiClientFactory**: Don't create direct API clients
2. **Handle errors gracefully**: Always check for API errors
3. **Add logging**: Use the logger service for debugging
4. **Test with mock data**: Use the provided mock responses
5. **Validate responses**: Always validate API response structure

## Submitting Changes

### Pull Request Process

1. **Create a feature branch** from `1.x`:
```bash
git checkout -b feature/your-feature-name
```

2. **Make your changes** following the coding standards

3. **Build the UI** (if applicable):
```bash
cd ui && npm run build
```

4. **Run tests** and ensure they pass:
```bash
ddev exec phpunit web/modules/contrib/acquia_optimize/tests
```

5. **Check coding standards**:
```bash
phpcs --standard=Drupal,DrupalPractice src/
```

6. **Commit your changes** with clear commit messages

7. **Push to your fork** and submit a pull request

### Pull Request Guidelines

- **Clear description**: Explain what your changes do and why
- **Link to issues**: Reference any related issue numbers
- **Screenshots**: Include screenshots for UI changes
- **Test coverage**: Ensure new code is covered by tests
- **Documentation**: Update documentation if needed

### Code Review Process

All contributions go through code review:
- Automated tests must pass
- Code must follow Drupal standards
- Security review for sensitive changes
- UI/UX review for interface changes
- Performance review for optimization changes

## Common Development Tasks

### Content Page Modal Development

When working on the content page integration:

1. **Form Integration**: Modify `AcquiaOptimizeFormAlter.php` for button behavior
2. **AJAX Handlers**: Update modal opening and response handling logic
3. **Modal Templates**: Edit `acquia-optimize-modal.html.twig` for UI changes
4. **JavaScript**: Modify `quickScan.js` for interaction behavior
5. **Styling**: Update CSS files in `css/` directory

### Modal Stage Development

The modal uses conditional rendering based on `data-stage` attribute:

- **stage="initial"**: Entry point with scan button and configuration link
- **stage="scanning"**: Progress indicator and status messages
- **stage="results"**: Comprehensive results with collapsible sections

### Canvas Extension Development

For Canvas integration changes:

1. **React Components**: Modify components in `ui/components/`
2. **Build Process**: Run `npm run build` after changes
3. **State Management**: Update component state in `ui/components/hooks/`
4. **API Integration**: Modify service calls in `ui/components/services/`
5. **Styling**: Update extension CSS for Canvas specific styling

### API Endpoint Development

When adding new API endpoints:

1. **Controller Methods**: Add methods to `AcquiaOptimizeController`
2. **Routing**: Update `acquia_optimize.routing.yml`
3. **Permissions**: Ensure proper permission checks
4. **Response Format**: Follow established JSON response patterns
5. **Error Handling**: Implement comprehensive error responses

### Template Development

When modifying the modal interface:

1. **Stage Structure**: Maintain the three-stage modal architecture
2. **Variable Documentation**: Update template header comments
3. **Accessibility**: Ensure ARIA labels and keyboard navigation
4. **Responsive Design**: Test across different screen sizes
5. **JavaScript Integration**: Coordinate with JS for interactive elements

## Debugging

### Debug Mode
Enable debug mode in the module configuration for detailed logging:
- API request/response logging in browser console
- Modal stage transition tracking
- Processing step tracking with timestamps
- Error details and stack traces for troubleshooting

### Common Issues & Solutions

#### Content Page Modal Issues
- **Modal Not Opening**: Check form validation errors and AJAX callback responses
- **Scan Button Disabled**: Verify node is saved and user has scan permissions
- **API Configuration Missing**: Ensure API key and URL are properly configured

#### Canvas Integration Issues
- **Extension Not Loading**: Check Canvas module compatibility and React build status
- **Scan Results Not Displaying**: Verify API responses in network tab
- **Permission Errors**: Confirm user has both Canvas and Web Governance permissions

#### API Connection Issues
- **Network Connectivity**: Check API URL accessibility and HTTPS requirements
- **Authentication Errors**: Verify API key validity and account status
- **Rate Limiting**: Monitor for API rate limit responses
- **CORS Issues**: Ensure proper domain configuration in API settings

### Development Debugging Tools

#### Browser DevTools
- **Network Tab**: Monitor AJAX requests and API responses
- **Console Tab**: View debug logs and JavaScript errors
- **Elements Tab**: Inspect modal DOM structure and CSS classes

#### Drupal Debugging
- **Database Logs**: Check `/admin/reports/dblog` for server-side errors
- **Configuration**: Verify module settings at `/admin/config/content/acquia-optimize`
- **Permissions**: Review user permissions at `/admin/people/permissions`

#### Modal State Debugging
- **Stage Tracking**: Monitor `data-stage` attribute changes in DOM
- **JavaScript Breakpoints**: Set breakpoints in `quickScan.js` for workflow debugging
- **AJAX Response Inspection**: Examine modal content updates in real-time

## Resources

- [Drupal Coding Standards](https://www.drupal.org/docs/develop/standards)
- [Canvas Documentation](https://www.drupal.org/project/canvas)
- [Drupal Testing Guide](https://www.drupal.org/docs/testing)

## Getting Help

- **Issue Queue**: [Project issue queue](https://www.drupal.org/project/issues/acquia_optimize)
- **Documentation**: Check the README.md for usage instructions
- **Maintainers**: Contact the module maintainers for complex issues
