<?php

namespace Drupal\acquia_optimize;

use Drupal\Core\Asset\LibraryDiscoveryInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;

/**
 * API client service.
 */
class ApiClient {

  use StringTranslationTrait;

  /**
   * Constructor.
   *
   * @param \GuzzleHttp\ClientInterface $client
   *   The Guzzle client.
   * @param \Drupal\Core\Asset\LibraryDiscoveryInterface $libraryDiscovery
   *   The library discovery service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger factory service.
   * @param string|null $apiKey
   *   The API key to use, if not set will use the one from config.
   * @param string|null $apiUrl
   *   The API Url to use, if not set will use the one from config.
   */
  public function __construct(
    private readonly ClientInterface $client,
    private readonly LibraryDiscoveryInterface $libraryDiscovery,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly LoggerInterface $logger,
    private ?string $apiKey = NULL,
    private ?string $apiUrl = NULL,
  ) {}

  /**
   * Validates API connection for web governance.
   *
   * @return array
   *   Validation result with success/error information.
   */
  public function validateApiConnection(): array {
    return $this->request('/account', 'GET');
  }

  /**
   * Create scan request.
   *
   * @param string $encodedPage
   *   The encoded page.
   * @param string $html
   *   The HTML to scan.
   * @param string $css
   *   The CSS to scan.
   * @param string $accessibility
   *   The accessibility level to use for the scan.
   *
   * @return array
   *   The scan request response.
   */
  public function createScanRequest(string $encodedPage, string $html, string $css, string $accessibility = ''): array {
    if ($accessibility === '') {
      $accessibility = $this->configFactory->get('acquia_optimize.settings')->get('accessibility') ?? "";
    }
    return $this->request('/html_scans', 'POST', [
      'encoded_page' => $encodedPage,
      'html' => $html,
      'css' => $css,
      'accessibility' => $accessibility,
    ]);
  }

  /**
   * Check scan status.
   *
   * @param string $scanId
   *   The scan ID.
   *
   * @return array
   *   The scan status response.
   */
  public function checkScanStatus(string $scanId): array {
    return $this->request('/html_scans/' . $scanId, 'GET');
  }

  /**
   * Request.
   *
   * @param string $endpoint
   *   Endpoint to call for the Acquia Web Governance API.
   * @param string $method
   *   HTTP method to use.
   * @param array $body
   *   Request body defaults to empty array.
   *
   * @return array
   *   Response or error object.
   */
  private function request(string $endpoint, string $method, array $body = []): array {
    $response = $this->validateLibrary();
    if ($response) {
      return $response;
    }

    $credentials = $this->getCredentials();

    $options = [
      'headers' => [
        'Authorization' => 'Bearer ' . $credentials['key'],
      ],
      'timeout' => 45,
      'connect_timeout' => 15,
      'read_timeout' => 30,
    ];
    if (!empty($body)) {
      $options['json'] = $body;
    }

    try {
      $response = $this->client->request($method, $credentials['url'] . $endpoint, $options);
      $status_code = $response->getStatusCode();
      $response_body = $response->getBody()->getContents();

      // Check for successful status codes.
      if ($status_code >= 200 && $status_code < 300) {
        return json_decode($response_body, TRUE, 512, JSON_THROW_ON_ERROR);
      }

      // Handle API error responses.
      return $this->handleApiError($status_code);
    }
    catch (RequestException $e) {
      // Handle HTTP errors with response.
      if ($e->hasResponse()) {
        $response = $e->getResponse();
        $status_code = $response->getStatusCode();

        return $this->handleApiError($status_code);
      }
      return $this->handleApiError(500);
    }
    catch (\Exception $e) {
      // Handle other errors (JSON decode, etc.).
      return $this->handleApiError(500);
    }
  }

  /**
   * Handles API error responses with simple status code mapping.
   *
   * @param int $statusCode
   *   The HTTP status code.
   *
   * @return array
   *   Simple error response with status and error message.
   */
  private function handleApiError(int $statusCode): array {
    // Map status codes to simple error messages.
    $errorMessage = match ($statusCode) {
      400 => 'Invalid request: Please check your input and try again.',
      401 => 'Authentication failure: Please check your API key.',
      403 => 'Access denied: You may not have permission to access this resource.',
      404 => 'The requested resource was not found.',
      422 => 'Access to the Acquia Web Governance API was denied: Contact your administrator for help.',
      429 => 'Too many requests: Please wait before trying again.',
      500 => 'Server error occurred: Please try again later.',
      502, 503 => 'Service temporarily unavailable. Please try again later.',
      504 => 'Request timeout: Please try again later.',
      default => 'An error occurred while processing your request.',
    };
    $errorMessage = $this->t('@error', ['@error' => $errorMessage]);

    $this->logger->error("$statusCode: $errorMessage");

    return [
      'status' => $statusCode,
      'error' => $errorMessage,
    ];
  }

  /**
   * Validates that the required library is installed.
   *
   * @return array|null
   *   Error array if validation fails, null if validation passes.
   */
  public function validateLibrary(): ?array {
    $path = $this->libraryDiscovery->getLibraryByName('acquia_optimize', 'optimize')['js'][0]['data'];
    if (!file_exists($path)) {
      return [
        'status' => 404,
        'error' => $this->t('The Acquia Web Governance library is not installed. Please obtain the library from Acquia and place it at <strong>@path</strong>.', ['@path' => $path]),
      ];
    }
    return NULL;
  }

  /**
   * Gets credentials from object properties or config.
   */
  private function getCredentials(): array {
    $optimize_settings = $this->configFactory->get('acquia_optimize.settings');
    return [
      'key' => $this->apiKey ?? $optimize_settings->get('api_key'),
      'url' => $this->apiUrl ?? $optimize_settings->get('api_url'),
    ];
  }

}
