import { useEffect } from 'react';
import { useSelector } from 'react-redux';
import { useConceptProverState } from './useConceptProverState';
import { useApiService } from '../services/apiService';
import { useFormHandlers } from './useFormHandlers';
import { useUIHandlers } from './useUIHandlers';
import { ScanService } from '../services/ScanService';
import { PollingService } from '../services/PollingService';
import { debugError } from '../utils/constants';

// Module-level variables that persist across component mounts/unmounts
let hasSettingsBeenFetched = false;
let cachedSettings = null;
let fetchPromise = null;

export const useConceptProverLogic = () => {
  const state = useConceptProverState();
  const apiService = useApiService(state.csrfToken);

  // Initialize services
  const scanService = new ScanService(apiService, state.isDebugMode);
  const pollingService = new PollingService(apiService, state.isDebugMode);

  // Initialize custom hooks
  const { handleSubmit } = useFormHandlers(state, apiService);
  const { toggleSection, showIssuesForLevel, showSeoIssuesForCategory } = useUIHandlers(state);

  // Get the layout from the Redux store
  const layout = useSelector((state) => state?.layoutModel?.present?.layout);

  // Helper function to apply settings to state
  const applySettingsToState = (data) => {
    state.setCanConfigure(!!data.can_configure);
    state.setCanScan(!!data.can_scan);

    // If user doesn't have permission to scan or configure, don't proceed further
    if (!data.can_scan && !data.can_configure) {
      state.setCurrentView('no-permission');
      return;
    }

    if (data.api_key) {
      state.setApiKey(data.api_key);
      // If we have an API key, show the quick scan view instead of initial
      state.setCurrentView('quick-scan');
    } else {
      // No API key, show initial view
      state.setCurrentView('initial');
    }

    if (data.api_url) {
      state.setApiUrl(data.api_url);
    }
    if (data.accessibility) {
      state.setAccessibilityGuideline(data.accessibility);
    }
    // Set debug mode from settings
    state.setIsDebugMode(!!data.debug_mode);
    
    // Set CSRF token from settings
    if (data.csrf_token) {
      state.setCsrfToken(data.csrf_token);
    }
  };

  // Fetch existing settings on component mount
  useEffect(() => {
    // If we already have cached settings, use them immediately
    if (hasSettingsBeenFetched && cachedSettings) {
      applySettingsToState(cachedSettings);
      state.setIsInitialLoading(false);
      return;
    }

    // If there's already a fetch in progress, wait for it
    if (fetchPromise) {
      fetchPromise
        .then((data) => {
          applySettingsToState(data);
        })
        .catch((error) => {
          state.setCurrentView('initial');
        })
        .finally(() => {
          state.setIsInitialLoading(false);
        });
      return;
    }

    // Start new fetch
    const fetchSettingsData = async () => {
      state.setIsInitialLoading(true);
      try {
        const data = await apiService.fetchSettings();

        // Cache the results
        hasSettingsBeenFetched = true;
        cachedSettings = data;
        fetchPromise = null;

        applySettingsToState(data);
        return data;
      } catch (error) {
        fetchPromise = null;
        state.setCurrentView('initial');
        throw error;
      } finally {
        state.setIsInitialLoading(false);
      }
    };

    // Store the promise to prevent duplicate fetches
    fetchPromise = fetchSettingsData();
  }, []);

  const startQuickScan = async (isRetry = false) => {
    // Check permission first
    if (!state.canScan && !state.canConfigure) {
      return;
    }

    try {
      state.setCurrentView('scanning');
      state.setScanProgress(0);

      // Use the scan service to start the scan
      const { scanId, progressInterval } = await scanService.startQuickScan(
        layout,
        state.setScanProgress,
        isRetry,
        state.accessibilityGuideline
      );

      state.setScanId(scanId);

      // Use the polling service to poll for results
      pollingService.pollScanResults(
        scanId,
        progressInterval,
        state.setScanProgress,
        state.setScanResults,
        state.setCurrentView
      );
    } catch (error) {
      debugError(state.isDebugMode, 'Error in startQuickScan:', error);

      // Auto-retry once if this isn't already a retry attempt
      if (!isRetry) {
        setTimeout(() => startQuickScan(true), 1000);
        return;
      }

      // Display error in results view after retry attempt failed
      state.setScanResults({
        status: 'failed',
        error: true,
        message: error.message || 'Failed to start quick scan'
      });

      setTimeout(() => {
        state.setCurrentView('results');
      }, 500);
    }
  };

  return {
    // State
    ...state,
    layout,

    // Handlers
    handleSubmit,
    startQuickScan,
    toggleSection,
    showIssuesForLevel,
    showSeoIssuesForCategory
  };
};

// Export utility functions for cache management
export const clearSettingsCache = () => {
  hasSettingsBeenFetched = false;
  cachedSettings = null;
  fetchPromise = null;
};

export const getSettingsCache = () => {
  return {
    hasSettingsBeenFetched,
    cachedSettings,
    hasFetchInProgress: !!fetchPromise
  };
};
