import { GenericCheckIcon, AccessibilityIcon, GenericWarningIcon } from '../Icons';

const AccessibilitySection = ({
  accessibilityIssues,
  accessibilityCount,
  toggleSection,
  expandedSections,
  detailedIssues,
  showIssuesForLevel
}) => {

  return (
    <div className="ao-category-section">
      <div
        className="ao-category-header"
        onClick={() => toggleSection('accessibility')}
      >
        <div className="ao-category-title">
          <AccessibilityIcon />
          <h3 className="ao-category-name">Accessibility</h3>
        </div>

        <div className="ao-category-count">
          {accessibilityCount > 0 ? (
            <>
              <GenericWarningIcon />
              <span className="ao-count-number ao-count-number--error">{accessibilityCount}</span>
            </>
          ) : (
            <>
              <GenericCheckIcon />
              <span className="ao-count-number ao-count-number--success">0</span>
            </>
          )}

          <svg
            className={`ao-category-arrow ${expandedSections.accessibility ? 'ao-category-arrow--expanded' : ''}`}
            viewBox="0 0 24 24"
            fill="#333"
          >
            <path d="M7 10L12 15L17 10H7Z"/>
          </svg>
        </div>
      </div>

      {expandedSections.accessibility && (
        <div className="ao-category-content">
          {accessibilityIssues.length === 0 ? (
            <div style={{
              padding: '20px',
              textAlign: 'center',
              color: '#666',
              fontSize: '14px',
              backgroundColor: '#f8f9fa',
              borderRadius: '4px',
              border: '1px solid #e9ecef'
            }}>
              No accessibility issues were detected on this page.
            </div>
          ) : (
            <>
              {/* Group issues by level A/AA */}
              {['A', 'AA', 'AAA'].map(level => {
                const levelIssues = accessibilityIssues.filter(issue => issue.level === level);
                if (levelIssues.length === 0) return null;

                return (
                  <div
                    key={level}
                    className="ao-subcategory"
                    onClick={(e) => showIssuesForLevel(level, e)}
                  >
                    <div className="ao-subcategory-label">
                      <div className={`ao-subcategory-badge ao-subcategory-badge--${level.toLowerCase()}`}>
                        {level}
                      </div>
                      <span className="ao-subcategory-text">Failing {level} checks</span>
                    </div>
                    <div className="ao-subcategory-count">
                      <span className="ao-subcategory-count-text">View {levelIssues.length} issues</span>
                      <svg
                        className={`ao-subcategory-arrow ${detailedIssues[level] ? 'ao-subcategory-arrow--expanded' : ''}`}
                        viewBox="0 0 24 24"
                        fill="currentColor"
                      >
                        <path d="M14 5l7 7-7 7V5z" />
                      </svg>
                    </div>
                  </div>
                );
              })}

              {/* Display detailed issues when expanded */}
              {['A', 'AA', 'AAA'].map(level => {
                const levelIssues = accessibilityIssues.filter(issue => issue.level === level);
                if (levelIssues.length === 0 || !detailedIssues[level]) return null;

                return (
                  <div key={`${level}-details`} className="ao-issues-container">
                    <h4 className="ao-issues-heading">
                      Level {level} Issues
                    </h4>

                    {levelIssues.map((issue, index) => (
                      <div key={index} className="ao-issue">
                        <div className="ao-issue-header">
                          <GenericWarningIcon />
                          <div className="ao-issue-title">
                            {issue.friendly_name || issue.name}
                          </div>
                        </div>

                        <div className="ao-issue-content">
                          { issue.description && issue.description !== 'N/A' && (
                            <p className="ao-issue-description">
                              {issue.description}
                            </p>
                          )}
                          {issue.error_source_code && (
                            <div className="ao-issue-code">
                              {issue.error_source_code}
                            </div>
                          )}
                          <div className="ao-issue-meta">
                            <span className="ao-issue-meta-label">Impact:</span> {issue.impact || 'Unknown'}
                            {issue.tags && issue.tags.length > 0 && (
                              <span style={{ marginLeft: '15px' }}>
                                <span className="ao-issue-meta-label">Affects:</span> {issue.tags.join(', ')}
                              </span>
                            )}
                          </div>
                        </div>
                      </div>
                    ))}
                  </div>
                );
              })}
            </>
          )}
        </div>
      )}
    </div>
  );
};

export default AccessibilitySection;
