(function (Drupal, drupalSettings, $) {
  'use strict';

  /**
   * Quick Scan functionality for Acquia Web Governance.
   */
  Drupal.behaviors.acquiaOptimizeQuickScan = {
    currentScanId: null,

    attach: function (context, settings) {
      var previewUrl = settings.acquiaOptimize && settings.acquiaOptimize.previewUrl ? settings.acquiaOptimize.previewUrl : null;

      if (!$('.acquia-optimize-modal-content').length && previewUrl !== null) {
        return;
      }

      $(document).off('click.ao-scan-button').on('click.ao-scan-button', '#ao-start-scan-btn, .acquia-optimize-retry-btn', function() {
        Drupal.behaviors.acquiaOptimizeQuickScan.startScan();
        return false;
      });

      this.initializeCollapsibles();
    },

    initializeCollapsibles: function() {
      $(document).off('click.categoryheader').on('click.categoryheader', '.ao-category-header', function(e) {
        var $header = $(this);
        var $section = $header.closest('.ao-category-section');
        var $content = $section.find('.ao-category-content').first();
        var isExpanded = $content.is(':visible');
        if (isExpanded) {
          $content.hide();
          $header.attr('aria-expanded', 'false');
        } else {
          $content.show();
          $header.attr('aria-expanded', 'true');
        }
        return false;
      });

      $(document).off('click.subcategory').on('click.subcategory', '.ao-subcategory', function(e) {
        e.preventDefault();
        var $subcategory = $(this);
        var level = $subcategory.data('level');
        var $issuesContainer = $subcategory.siblings('.ao-issues-container[data-level-details="' + level + '"]');
        if ($issuesContainer.length) {
          var isVisible = $issuesContainer.is(':visible');
          if (isVisible) {
            $issuesContainer.hide();
          } else {
            $issuesContainer.show();
          }
        }
      });

      $(document).off('click.seocategory').on('click.seocategory', '.ao-seo-category-header', function(e) {
        e.preventDefault();
        var $header = $(this);
        var $seoCategory = $header.closest('.ao-seo-category');
        var category = $seoCategory.data('category');
        var $issuesContainer = $seoCategory.find('.ao-seo-issues-container[data-category-details="' + category + '"]');
        if ($issuesContainer.length) {
          var isVisible = $issuesContainer.is(':visible');
          if (isVisible) {
            $issuesContainer.hide();
          } else {
            $issuesContainer.show();
          }
        }
      });

      $(document).off('click.rescan').on('click.rescan', '.acquia-optimize-rescan-btn', function(e) {
        e.preventDefault();
        Drupal.behaviors.acquiaOptimizeQuickScan.startScan();
      });
    },

    startScan: function () {
      var self = this;
      var modalContent = document.querySelector('.acquia-optimize-modal-content');
      if (!modalContent) {
        return false;
      }

      self.showScanningStage(modalContent);

      var progressBar = modalContent.querySelector('.acquia-optimize-progress-fill');
      var statusElement = modalContent.querySelector('.acquia-optimize-scanning-status p');

      if (typeof window.DataCollector === 'undefined') {
        self.showError(modalContent, Drupal.t('DataCollector library not loaded. Please check your setup.'));
        return false;
      }

      var settings = drupalSettings.acquiaOptimize || {};
      var previewUrl = settings.previewUrl;

      if (!previewUrl) {
        self.showError(modalContent, Drupal.t('No preview URL available. Please check your configuration.'));
        return false;
      }

      self.updateProgress(progressBar, statusElement, 25, Drupal.t('Fetching page content...'));

      $.ajax({
        url: previewUrl,
        method: 'GET',
        xhrFields: { withCredentials: true },
        success: function(htmlContent) {
          self.updateProgress(progressBar, statusElement, 50, Drupal.t('Processing content...'));
          try {
            var parser = new window.DOMParser();
            var doc = parser.parseFromString(htmlContent, 'text/html');
            var dataCollector = new window.DataCollector();
            dataCollector.collectData(doc.documentElement).then(function(collectedData) {
              if (!collectedData || !collectedData.dom_tree || !collectedData.html) {
                throw new Error(Drupal.t('Invalid data collected from DataCollector'));
              }
              var encodedPageData = {
                dom_tree: collectedData.dom_tree,
                css: collectedData.css,
                version: '4.1.3',
                viewport: collectedData.viewport
              };
              var scanRequest = {
                encoded_page: JSON.stringify(encodedPageData),
                html: collectedData.html,
                css: collectedData.css.join('\n'),
                accessibility: settings.accessibility || 'WCAG21-AA'
              };
              self.getContentQuickScan(scanRequest, progressBar, statusElement, function(result) {
                if (result.status === 'success' || result.status === 'completed') {
                  self.showRenderedResults(modalContent, result.rendered_results);
                  self.currentScanId = result.scan_id;
                } else {
                  self.showRenderedResults(modalContent, result.rendered_results);
                }
              });
            }).catch(function(dataError) {
              self.showError(modalContent, Drupal.t('Failed to process page data: %message', {'%message': dataError.message}));
            });
          } catch (e) {
            self.showError(modalContent, Drupal.t('Failed to process page data: %message', {'%message': e.message}));
          }
        },
        error: function(jqXHR) {
          self.showError(modalContent, 'HTTP error! status: ' + jqXHR.status);
        }
      });

      return false;
    },

    getContentQuickScan: function(scanRequest, progressBar, statusElement, callback) {
      var self = this;
      self.updateProgress(progressBar, statusElement, 75, Drupal.t('Submitting scan request...'));
      $.ajax({
        url: '/acquia-optimize/api/quick-scan',
        method: 'POST',
        contentType: 'application/json',
        headers: { 'X-Requested-With': 'XMLHttpRequest' },
        data: JSON.stringify(scanRequest),
        success: function(createResult) {
          if (createResult.status === 'error' || !createResult.id) {
            var errorMsg = createResult.message || createResult.error || Drupal.t('Failed to create scan');
            callback({ status: 'error', message: errorMsg, rendered_results: null });
            return;
          }
          var scanId = createResult.id;
          self.updateProgress(progressBar, statusElement, 80, Drupal.t('Processing scan (server-side polling)...'));
          $.ajax({
            url: '/acquia-optimize/api/content-quick-scan/' + scanId,
            method: 'GET',
            headers: { 'X-Requested-With': 'XMLHttpRequest' },
            xhrFields: { withCredentials: true },
            success: function(finalResult) {
              self.updateProgress(progressBar, statusElement, 100, Drupal.t('Finalizing results...'));
              if (finalResult.status === 'failed' || finalResult.error_message) {
                callback({
                  status: 'failed',
                  message: finalResult.error_message || finalResult.message || Drupal.t('Scan failed'),
                  rendered_results: finalResult.rendered_results
                });
                return;
              }
              callback({
                status: 'success',
                scan_id: scanId,
                rendered_results: finalResult.rendered_results
              });
            },
            error: function(jqXHR) {
              callback({
                status: 'error',
                message: Drupal.t('HTTP %status: Failed to get scan results', {'%status': jqXHR.status}),
                rendered_results: null
              });
            }
          });
        },
        error: function(jqXHR) {
          callback({
            status: 'error',
            message: Drupal.t('HTTP %status: Failed to create scan', {'%status': jqXHR.status}),
            rendered_results: null
          });
        }
      });
    },

    updateProgress: function(progressBar, statusElement, percent, message) {
      if (progressBar) {
        progressBar.style.width = percent + '%';
      }
      if (statusElement) {
        statusElement.textContent = message;
      }
    },

    showScanningStage: function (modalContent) {
      if (modalContent) {
        modalContent.setAttribute('data-stage', 'scanning');
        var progressBar = modalContent.querySelector('.acquia-optimize-progress-fill');
        if (progressBar) {
          progressBar.style.width = '10%';
        }
        var statusElement = modalContent.querySelector('.acquia-optimize-scanning-status p');
        if (statusElement) {
          statusElement.textContent = Drupal.t('Scanning in-progress...');
        }
      }
    },

    showError: function (modalContent, message) {
      if (modalContent) {
        modalContent.setAttribute('data-stage', 'results');
        var resultsStage = modalContent.querySelector('.stage-results');
        if (resultsStage) {
          resultsStage.innerHTML =
            '<div class="acquia-optimize-error">' +
              '<p><strong>' + Drupal.t('Error:') + '</strong> ' + message + '</p>' +
              '<button class="acquia-optimize-retry-btn" data-action="retry-scan">' + Drupal.t('Retry Scan') + '</button>' +
            '</div>';
        }
      }
    },

    showRenderedResults: function(modalContent, renderedHtml) {
      var self = this;
      if (modalContent && renderedHtml) {
        modalContent.outerHTML = renderedHtml;
        setTimeout(function() {
          self.initializeCollapsibles();
        }, 100);
      } else if (!renderedHtml) {
        self.showError(modalContent, Drupal.t('No results received from server'));
      }
    }
  };

})(Drupal, drupalSettings, jQuery);
