<?php

declare(strict_types=1);

namespace Drupal\acquia_optimize\Controller;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Access\CsrfTokenGenerator;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\acquia_optimize\ApiClientFactory;
use Drupal\acquia_optimize\Enum\ReadabilityData;
use Drupal\acquia_optimize\Enum\SeoIssues;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Psr\Log\LoggerInterface;

/**
 * Controller for Acquia Web Governance API endpoints.
 */
class AcquiaOptimizeController extends ControllerBase {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The API client service.
   *
   * @var \Drupal\acquia_optimize\ApiClientFactory
   */
  protected $apiClientFactory;

  /**
   * Logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The CSRF token generator.
   *
   * @var \Drupal\Core\Access\CsrfTokenGenerator
   */
  protected $csrfTokenGenerator;

  /**
   * Constructs a new AcquiaOptimizeController object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $current_user
   *   The current user.
   * @param \Drupal\acquia_optimize\ApiClientFactory $api_client_factory
   *   The API client service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger factory service.
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token_generator
   *   The CSRF token generator.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountProxyInterface $current_user,
    ApiClientFactory $api_client_factory,
    LoggerInterface $logger,
    CsrfTokenGenerator $csrf_token_generator,
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->apiClientFactory = $api_client_factory;
    $this->logger = $logger;
    $this->csrfTokenGenerator = $csrf_token_generator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('current_user'),
      $container->get('acquia_optimize.api_client_factory'),
      $container->get('logger.channel.acquia_optimize'),
      $container->get('csrf_token')
    );
  }

  /**
   * Gets the Acquia Web Governance settings.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response containing settings.
   */
  public function getSettings(): JsonResponse {
    try {
      $config = $this->configFactory->get('acquia_optimize.settings');
      $hasAdminPermission = $this->currentUser()->hasPermission('administer acquia optimize');
      $hasScanPermission = $this->currentUser()->hasPermission('scan acquia optimize');

      $settings = [
        'api_key' => $config->get('api_key') ? '[API key set]' : '',
        'api_url' => $config->get('api_url') ?? '',
        'accessibility' => $config->get('accessibility') ?? 'WCAG21-AA',
        'can_configure' => $hasAdminPermission,
        'can_scan' => $hasScanPermission,
        'debug_mode' => $config->get('debug_mode') ?? FALSE,
        'csrf_token' => $this->csrfTokenGenerator->get('acquia_optimize'),

      ];

      return new JsonResponse($settings);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to retrieve Acquia Optimize settings: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse(['status' => 'error', 'message' => 'Failed to retrieve settings'], 500);
    }

  }

  /**
   * Updates the Acquia Optimize settings.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response indicating success or failure.
   */
  public function updateSettings(Request $request): JsonResponse {
    try {
      $data = json_decode($request->getContent(), TRUE, 512, JSON_THROW_ON_ERROR);

      $config = $this->configFactory->getEditable('acquia_optimize.settings');

      if (isset($data['api_key'])) {
        $config->set('api_key', $data['api_key']);
      }

      if (isset($data['api_url'])) {
        $config->set('api_url', $data['api_url']);
      }

      if (isset($data['accessibility'])) {
        $config->set('accessibility', $data['accessibility']);
      }

      $config->save();

      return new JsonResponse(['status' => 'success', 'message' => 'Settings updated successfully']);
    }
    catch (\Exception $e) {
      return new JsonResponse(['status' => 'error', 'message' => $e->getMessage()], 400);
    }
  }

  /**
   * Creates a quick scan.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function createQuickScan(Request $request): JsonResponse {
    try {
      $content = $request->getContent();
      ['encoded_page' => $encoded_page, 'html' => $html, 'css' => $css, 'accessibility' => $accessibility] = Json::decode($content);
      $api_client_factory = $this->apiClientFactory->create();
      $response = $api_client_factory->createScanRequest($encoded_page, $html, $css, $accessibility);
      return new JsonResponse($response);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create quick scan: @message', ['@message' => $e->getMessage()]);
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Failed to create scan. Please try again.',
      ], 500);
    }
  }

  /**
   * Checks the status of a scan.
   *
   * @param string $scan_id
   *   The scan ID.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The response.
   */
  public function getQuickScan(string $scan_id): JsonResponse {
    try {
      $api_client_factory = $this->apiClientFactory->create();
      $response = $api_client_factory->checkScanStatus($scan_id);
      if ($response['status'] === 'failed') {
        $error = $response['failure_reason'] ?? $response['error_message'] ?? "for scan id $scan_id";
        $this->logger->error('Scan failed: @message', ['@message' => $error]);
        return new JsonResponse($response);
      }
      $processed_response = $this->processScanResults($response);
      return new JsonResponse($processed_response);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get scan status for scan ID @scan_id: @message', [
        '@scan_id' => $scan_id,
        '@message' => $e->getMessage(),
      ]);
      return new JsonResponse([
        'status' => 'error',
        'message' => 'Failed to retrieve scan status. Please try again.',
      ], 500);
    }
  }

  /**
   * Validates API connection using provided credentials.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object containing API credentials.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response with validation result.
   */
  public function validateConnection(Request $request): JsonResponse {
    try {
      $data = json_decode($request->getContent(), TRUE, 512, JSON_THROW_ON_ERROR);

      if (!isset($data['api_key']) || !isset($data['api_url'])) {
        return new JsonResponse([
          'status' => 400,
          'error' => 'Missing required parameters: api_key and api_url',
        ], 400);
      }

      // Validate the connection using the API client.
      $api_client_factory = $this->apiClientFactory->create($data['api_key'], $data['api_url']);
      $result = $api_client_factory->validateApiConnection();

      return new JsonResponse($result);
    }
    catch (\Exception $e) {

      return new JsonResponse([
        'status' => 500,
        'error' => 'API connection failed. Please check your API credentials and try again.',
      ], 500);
    }
  }

  /**
   * Process scan results data and return template variables.
   *
   * @param array $result_data
   *   The raw result data from the scan.
   *
   * @return array
   *   Refined scan results.
   */
  private function processScanResults(array $result_data): array {
    // Extract data from the successful response.
    $data = $result_data['scan_result'] ?? $result_data;

    // Get individual issue arrays with fallbacks.
    $seo_issues = $data['seo_issues'] ?? [];
    $accessibility_issues = $data['accessibility_errors'] ?? [];
    $data_protection_violations = $data['data_protection_violations'] ?? [];

    // Process SEO issues to add friendly names and categories.
    $processed_seo_issues = SeoIssues::getGroupedSeoIssues($seo_issues);

    // Process readability data.
    $readability = ReadabilityData::fromScore($data['readability'] ?? 0);

    $readability_data = [
      'score_display' => $data['readability'] ?? 0,
      'grade_level' => $readability->getGradeLevel()->render(),
      'description' => $readability->getDescription()->render(),
      'score_color' => $readability->getColor(),
    ];

    $total_issues = count($seo_issues) + count($accessibility_issues) +
      count($data_protection_violations);

    $process_results = [
      'total_issues' => $total_issues,
      'scan_date' => date('F j, Y - g:i A'),
      'accessibility_errors' => $accessibility_issues,
      'seo_issues' => $processed_seo_issues,
      'readability_data' => $readability_data,
      'data_protection_violations' => $data_protection_violations,
    ];
    $output = [];

    $output = [
      'scan_result' => $result_data['scan_result'] ?? [],
      'status' => $result_data['status'],
    ];

    if ($result_data['status'] === 'completed') {
      $output['process_result'] = $process_results;
    }

    return $output;
  }

}
