<?php

declare(strict_types=1);

namespace Drupal\awg_auth_test\HttpClientMiddleware;

use Drupal\Core\Cache\CacheBackendInterface;
use GuzzleHttp\Promise\FulfilledPromise;
use GuzzleHttp\Promise\PromiseInterface;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\RequestInterface;
use Psr\Log\LoggerInterface;

/**
 * Intercepts API requests to the Acquia Web Governance to return mocked data.
 */
final class MockedResponseMiddleware {

  private const CACHE_KEY = 'awg_auth_test.request_counter';

  const TEST_API_KEY = '0123456789abcdefghiklmnopqrstuvwxyz';

  /**
   * Array to track AJAX request counts.
   *
   * @var array
   */
  private array $ajaxRequestCounts = [];

  /**
   * Constructs a new MockedResponseMiddleware object.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend.
   */
  public function __construct(
    private readonly LoggerInterface $logger,
    private readonly CacheBackendInterface $cache,
  ) {
  }

  /**
   * Gets the current request counter from cache.
   *
   * @return int
   *   The request counter value.
   */
  private function getRequestCounter(): int {
    $item = $this->cache->get(self::CACHE_KEY);
    return $item ? (int) $item->data : 1;
  }

  /**
   * Sets the request counter in cache.
   *
   * @param int $value
   *   The counter value to set.
   */
  private function setRequestCounter(int $value): void {
    $this->cache->set(self::CACHE_KEY, $value);
  }

  /**
   * Added a middleware that intercepts requests and returns mocked responses.
   */
  public function __invoke(): callable {
    return function (callable $handler): callable {
      return function (RequestInterface $request, array $options) use ($handler): PromiseInterface {
        if ($request->getUri()->getHost() === 'example.monsido.com') {
          $fixturesDirectory = $this->getFixturesDirectory();
          $path = $request->getUri()->getPath();
          $authorization = $request->getHeader('Authorization');
          $authorization = $authorization[0] ?? [];
          if (!$authorization || !str_contains($authorization, 'Bearer')) {
            return new FulfilledPromise(
              new Response(
                401,
                ['Content-Type' => 'application/json'],
                file_get_contents($fixturesDirectory . '/missing_authorization.json'),
              ));
          }
          $apiKey = str_replace('Bearer ', '', $authorization);
          if ($apiKey !== self::TEST_API_KEY) {
            return new FulfilledPromise(
              new Response(
                401,
                ['Content-Type' => 'application/json'],
                file_get_contents($fixturesDirectory . '/invalid_token.json'),
              ));
          }
          if ($path === '/api/account') {
            $baseDirectory = $fixturesDirectory . '/account';
            return new FulfilledPromise(
              new Response(
                200,
                ['Content-Type' => 'application/json'],
                file_get_contents($baseDirectory . '/valid_account.json'),
              ));
          }
          elseif (str_starts_with($path, '/api/html_scans')) {
            $baseDirectory = $fixturesDirectory . '/html_scans';
            if ($request->getMethod() === 'POST') {
              $count = $this->getRequestCounter();
              $this->logger->info('Creating scan request number @count', ['@count' => $count]);
              $this->setRequestCounter($count + 1);
              return new FulfilledPromise(
                new Response(
                  200,
                  ['Content-Type' => 'application/json'],
                  file_get_contents($baseDirectory . "/{$count}_create_scan.json"),
                )
              );
            }
            elseif ($request->getMethod() === 'GET') {
              $scan_id = basename($path);
              $baseDirectory .= "/{$scan_id}";
              if (!file_exists($baseDirectory)) {
                throw new \Exception("Missing fixture directory for scan ID: {$scan_id}");
              }

              // Track the number of times this path is requested.
              if (!isset($this->ajaxRequestCounts[$path][$request->getMethod()][$scan_id])) {
                $this->ajaxRequestCounts[$path][$request->getMethod()][$scan_id] = 0;
              }
              $count = ++$this->ajaxRequestCounts[$path][$request->getMethod()][$scan_id];
              return new FulfilledPromise(
                new Response(
                  200,
                  ['Content-Type' => 'application/json'],
                  file_get_contents($baseDirectory . "/{$count}_response.json"),
                )
              );
            }
          }
        }
        return new FulfilledPromise(new Response(404));
      };
    };
  }

  /**
   * Gets the fixtures directory path.
   *
   * @return string
   *   The path to the fixtures directory.
   */
  private function getFixturesDirectory(): string {
    return dirname(__DIR__, 2) . '/fixtures';
  }

}
