import Button from '../Button';
import Footer from '../Footer';
import { AcquiaOptimizeLogo, RetryIcon } from '../Icons';
import { getDrupalSettings } from '../utils/constants';
import { ApiError, getErrorMessage, isRetryableError } from '../services/apiService';
import ResultsHeader from '../sections/ResultsHeader';
import ReadabilitySection from '../sections/ReadabilitySection';
import DataPrivacySection from '../sections/DataPrivacySection';
import AccessibilitySection from '../sections/AccessibilitySection';
import SEOSection from '../sections/SEOSection';



const ResultsView = ({
  scanResults,
  startQuickScan,
  canScan,
  canConfigure,
  expandedSections,
  toggleSection,
  detailedIssues,
  showIssuesForLevel,
  showSeoIssuesForCategory,
  apiError // Add this prop to handle API errors from parent component
}) => {

  // Enhanced error rendering with API service error handling
  const renderApiError = (error) => {
    const { imgSrc } = getDrupalSettings();
    const imgPath = imgSrc.replace(/\/[^\/]*$/, '');

    let errorMessage = 'An error occurred during the scan.';
    let statusCode = null;
    let isRetryable = false;
    let showSettingsLink = false;

    // Handle different error types
    if (error instanceof ApiError) {
      errorMessage = error.message;
      statusCode = error.statusCode;
      isRetryable = error.isRetryable;

      // Show settings link for authentication errors
      showSettingsLink = statusCode === 401 || statusCode === 403;
    } else if (typeof error === 'object' && error !== null) {
      // Handle scan result error objects
      errorMessage = error.failure_reason || error.message || error.error_message || error.error || errorMessage;

      // Extract status code if available
      if (error.status_code || error.statusCode) {
        statusCode = error.status_code || error.statusCode;
        errorMessage = getErrorMessage(statusCode, errorMessage);
        isRetryable = isRetryableError(statusCode);
        showSettingsLink = statusCode === 401 || statusCode === 403;
      }

      // Special handling for token/authentication errors
      if (errorMessage.includes('token must be passed') || errorMessage.includes('Authentication failed')) {
        statusCode = 401;
        errorMessage = getErrorMessage(401);
        showSettingsLink = true;
        isRetryable = false;
      }
    } else if (typeof error === 'string') {
      errorMessage = error;
    }

    // Special UI for authentication errors
    if (showSettingsLink) {
      return (
        <div style={{
          padding: '20px',
          textAlign: 'center',
          backgroundColor: '#ffeeee',
          borderRadius: '8px',
          color: '#cc0000',
          border: '1px solid #ffcccc'
        }}>
          <AcquiaOptimizeLogo OptimizeLogo={`${imgPath}/error.svg`} alt='Authentication Error' className='ao-scan' />
          <h3 style={{ margin: '15px 0 10px 0', color: '#cc0000' }}>Authentication Failed</h3>
          <p style={{ margin: '10px 0' }}>{errorMessage}</p>
          {statusCode && (
            <p style={{ fontSize: '0.9em', color: '#666', margin: '5px 0' }}>
              Error Code: {statusCode}
            </p>
          )}
          <p style={{ marginTop: '15px' }}>
            <a
              href="/admin/config/content/acquia-optimize"
              target="_blank"
              rel="noopener noreferrer"
              style={{
                color: '#0078ff',
                textDecoration: 'none',
                fontWeight: '500',
                padding: '8px 16px',
                border: '1px solid #0078ff',
                borderRadius: '4px',
                display: 'inline-block'
              }}
            >
              Go to Settings Page
            </a>
          </p>
        </div>
      );
    }

    // General error display
    return (
      <div className="ao-message ao-message--error" style={{
        margin: '20px 0',
        padding: '20px',
        display: 'flex',
        flexDirection: 'column',
        alignItems: 'center',
        gap: '15px',
        width: '100%',
        border: '1px solid #ffcccc',
        borderRadius: '8px'
      }}>
        <AcquiaOptimizeLogo OptimizeLogo={`${imgPath}/error.svg`} alt='Scan Error' className='ao-scan' />

        <div style={{ textAlign: 'center' }}>
          <h3 style={{ margin: '0 0 10px 0', color: '#cc0000' }}>
            {isRetryable ? 'Temporary Error' : 'Scan Failed'}
          </h3>
          <p style={{ margin: '10px 0', fontSize: '16px' }}>{errorMessage}</p>

          {statusCode && (
            <p style={{ fontSize: '0.9em', color: '#666', margin: '5px 0' }}>
              Error Code: {statusCode}
            </p>
          )}

          {isRetryable && (
            <p style={{ fontSize: '0.9em', color: '#666', margin: '10px 0' }}>
              This appears to be a temporary issue. Please try again.
            </p>
          )}
        </div>

        {(canScan || canConfigure) && (
          <Button
            onClick={startQuickScan}
            disabled={!canScan && !canConfigure}
            className="ao-button ao-button--icon"
            style={{
              backgroundColor: isRetryable ? '#28a745' : '#6c757d',
              borderColor: isRetryable ? '#28a745' : '#6c757d'
            }}
          >
            <RetryIcon />
            {isRetryable ? 'Try Again' : 'Retry Scan'}
          </Button>
        )}
      </div>
    );
  };

  // Parse and display scan results based on the provided screenshots
  const renderResults = () => {
    // Handle API errors from parent component (e.g., scan status check errors)
    if (apiError) {
      return renderApiError(apiError);
    }

    if (!scanResults) {
      return (
        <div style={{ padding: '20px', textAlign: 'center' }}>
          No results to display.
        </div>
      );
    }

    // Handle failed scans with enhanced error handling
    if (scanResults.status === 'failed' || scanResults.error) {
      return renderApiError(scanResults);
    }

    // Process accessibility issues
    const accessibilityIssues = scanResults.accessibility_errors || [];
    const accessibilityCount = accessibilityIssues.length;

    // Process SEO issues
    const seoIssues = scanResults.seo_issues || [];
    const seoCount = seoIssues.length;

    // Process data privacy issues
    const dataPrivacyIssues = scanResults.data_protection_violations || [];

    // Determine total issues count
    const totalIssues = accessibilityCount + seoCount + dataPrivacyIssues.length;

    // Scan result overall summary
    return (
      <>
       {/* Scan Results summary */}
        <ResultsHeader
          totalIssues={totalIssues}
          startQuickScan={startQuickScan}
          canScan={canScan}
          canConfigure={canConfigure}
        />

        {/* Accessibility Section */}
        <AccessibilitySection
          accessibilityIssues={accessibilityIssues}
          accessibilityCount={accessibilityCount}
          toggleSection={toggleSection}
          expandedSections={expandedSections}
          detailedIssues={detailedIssues}
          showIssuesForLevel={showIssuesForLevel}
        />

        {/* Readability Section */}
        <ReadabilitySection
          readabilityData={scanResults.readability_data}
          toggleSection={toggleSection}
          expandedSections={expandedSections}
        />

        {/* SEO Section */}
        <SEOSection
          seoIssues={seoIssues}
          seoCount={seoCount}
          toggleSection={toggleSection}
          expandedSections={expandedSections}
          detailedIssues={detailedIssues}
          showSeoIssuesForCategory={showSeoIssuesForCategory}
        />

        {/* Data Privacy Section */}
        <DataPrivacySection
          expandedSections={expandedSections}
          toggleSection={toggleSection}
          dataPrivacyIssues={dataPrivacyIssues}
        />
      </>
    );
  };

  return (
    <div className="ao-container">
      <div className="ao-results-container">
        {renderResults()}
        <Footer AcquiaDocs={true} />
      </div>
    </div>
  );
};

export default ResultsView;
