<?php

namespace Drupal\acsf_module_listing\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * ACSF Environment Entity form.
 *
 * @property \Drupal\acsf_module_listing\AcsfEnvironmentEntityInterface $entity
 */
class ACSFEnvironmentEntityForm extends EntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    $acsf_environment = $this->entity;

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $acsf_environment->label(),
      '#description' => $this->t('Label for the acsf environment entity.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $acsf_environment->id(),
      '#machine_name' => [
        'exists' => '\Drupal\acsf_module_listing\Entity\AcsfEnvironmentEntity::load',
      ],
      '#disabled' => !$this->entity->isNew(),
    ];

    $form['ssh_user'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SSH Server User'),
      '#maxlength' => 255,
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('ssh_user') : '',
      '#required' => TRUE,
    ];

    $form['ssh_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SSH Server URL'),
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('ssh_url') : '',
      '#required' => TRUE,
    ];

    $form['drush_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Drush Path'),
      '#description' => $this->t('Full path to the folder where drush command are available'),
      '#maxlength' => 255,
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('drush_path') : '',
      '#required' => TRUE,
    ];

    $form['acsf_env_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ACSF Environment URL'),
      '#maxlength' => 255,
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('acsf_env_url') : '',
      '#required' => TRUE,
      '#description' => $this->t('For example, https://www.dev.acquia.com'),
      '#element_validate' => [
        [$this, 'removeTrailingSlash'],
      ],
    ];

    $form['acsf_username'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ACSF Username'),
      '#maxlength' => 255,
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('acsf_username') : '',
      '#required' => TRUE,
    ];

    $form['acsf_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ACSF API Key'),
      '#maxlength' => 255,
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('acsf_api_key') : '',
      '#required' => TRUE,
    ];

    $form['public_key_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Public key path'),
      '#maxlength' => 255,
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('public_key_path') : '',
      '#required' => TRUE,
      '#description' => $this->t('For example, /home/username/.ssh/id_rsa.pub'),
    ];

    $form['private_key_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Private key path'),
      '#maxlength' => 255,
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('private_key_path') : '',
      '#required' => TRUE,
      '#description' => $this->t('For example, /home/username/.ssh/id_rsa'),
    ];

    $form['pass_phrase'] = [
      '#type' => 'textfield',
      '#title' => $this->t('SSH Key Pass Phrase'),
      '#maxlength' => 255,
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->getVariable('pass_phrase') : '',
      '#required' => FALSE,
      '#description' => $this->t('Optional. Only if the passcode is set.'),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->status() : TRUE,
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => !$acsf_environment->isNew() ? $acsf_environment->get('description') : '',
      '#description' => $this->t('Description of the acsf environment entity.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $message = $result == SAVED_NEW
      ? $this->t('Created new acsf environment entity %label.', $message_args)
      : $this->t('Updated acsf environment entity %label.', $message_args);
    $this->messenger()->addStatus($message);
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);
    $public_key = $form_state->getValue('public_key_path');
    $private_key = $form_state->getValue('private_key_path');

    if (!file_exists($public_key)) {
      $form_state->setError($form['public_key_path'], $this->t('Please, validate public file is exists and available.'));
    }

    if (!file_exists($private_key)) {
      $form_state->setError($form['private_key_path'], $this->t('Please, validate private file is exists and available.'));
    }
  }

  /**
   * Removes trailing slashes from the URL-based fields.
   *
   * @param $element
   *   Form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   *
   * @return void
   */
  public function removeTrailingSlash(&$element, FormStateInterface $form_state): void {
    $element_value = $element['#value'];
    $new_value = rtrim($element_value, '/');
    $form_state->setValueForElement($element, $new_value);
  }

}
