<?php

namespace Drupal\acsf_module_listing\Services;

use Drupal\acsf_module_listing\Entity\AcsfEnvironmentEntity;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use \Drupal\Core\Logger\LoggerChannelInterface;
use GuzzleHttp\ClientInterface;

/**
 * The class is responsible for REST API requests to Acquia Cloud API.
 */
class AcsfApiV1Service {

  /**
   * HTTP client service.
   *
   * @var \GuzzleHttp\ClientInterface
   *   HTTP client service variable.
   */
  protected ClientInterface $httpClient;

  /**
   * Messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   *   Messenger service variable.
   */
  protected MessengerInterface $messenger;

  /**
   * Logger channel service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   *   Logger channel service variable.
   */
  protected LoggerChannelInterface $logger;

  /**
   * AcsfApiV1Service constructor.
   *
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   Http client service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   Messenger service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   Logger channel factory service.
   */
  public function __construct(ClientInterface $httpClient, MessengerInterface $messenger, LoggerChannelFactoryInterface $loggerChannelFactory) {
    $this->httpClient = $httpClient;
    $this->messenger = $messenger;
    $this->logger = $loggerChannelFactory->get('acsf_module_listing');
  }

  /**
   * Processes request to ACSF instance.
   *
   * @param \Drupal\acsf_module_listing\Entity\AcsfEnvironmentEntity $acsf_environment
   *   Acsf Environment Entity.
   *
   * @return array|null
   *   Returns array of websites or NULL
   */
  public function getWebsitesList(AcsfEnvironmentEntity $acsf_environment): array|null {
    $url = $acsf_environment->getVariable('acsf_env_url') . '/api/v1/sites';
    $params['auth'] = [
      $acsf_environment->getVariable('acsf_username'),
      $acsf_environment->getVariable('acsf_api_key'),
    ];
    $params['query'] = ['limit' => 10000];

    try {
      $acsf_response = $this->httpClient->request('GET', $url, $params);
      if ($acsf_response->getStatusCode() === 200) {
        return json_decode($acsf_response->getBody()->getContents(), TRUE);
      }
      $this->logger->error('Request to the @url returns @code HTTP error', [
        '@url' => $url,
        '@code' => $acsf_response->getStatusCode(),
      ]);
    }
    catch (\Exception $exception) {
      $this->logger->error($exception->getMessage(), ['exception' => $exception]);
    }
    $this->messenger->addError('Something went wrong. Please check database logs.');
    return NULL;
  }

}
