<?php

namespace Drupal\acsf_module_listing\Form;

use Drupal\acsf_module_listing\Services\AcsfApiV1Service;
use Drupal\acsf_module_listing\Services\AcsfEnvironmentEntityService;
use Drupal\acsf_module_listing\Services\SshShellService;
use Drupal\Component\Utility\Html;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Implements the ACSFModulesListingForm.
 *
 * @see \Drupal\Core\Form\ConfigFormBase
 */
class ACSFModulesListingForm extends FormBase {

  /**
   * Config service.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   *   Config service variable.
   */
  protected ImmutableConfig $config;

  /**
   * Batch service.
   *
   * @var \Drupal\Core\Batch\BatchBuilder
   *   Batch service variable.
   */
  protected $batchBuilder;

  /**
   * Module list service.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   *   Module list service variable.
   */
  protected ModuleExtensionList $moduleList;

  /**
   * Cache service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   *   Cache service variable.
   */
  protected $cache;

  /**
   * ACSF environment entity service.
   *
   * @var \Drupal\acsf_module_listing\Services\AcsfEnvironmentEntityService
   *   ACSF environment entity service variable.
   */
  protected AcsfEnvironmentEntityService $acsfEnvironmentEntityService;

  /**
   * ACSF API V1 service.
   *
   * @var \Drupal\acsf_module_listing\Services\AcsfApiV1Service
   *   ACSF API V1 service variable.
   */
  protected AcsfApiV1Service $acsfApiV1Service;

  /**
   * SSH shell service.
   *
   * @var \Drupal\acsf_module_listing\Services\SshShellService
   *   SSH shell service variable.
   */
  protected SshShellService $sshShellService;

  /**
   * ACSFModulesListingForm constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   Config factory service variable.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   Cache service variable.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   Request stack service variable.
   * @param \Drupal\Core\Extension\ModuleExtensionList $moduleList
   *   Module list service variable.
   * @param \Drupal\acsf_module_listing\Services\AcsfEnvironmentEntityService $acsfEnvironmentEntityService
   *   ACSF Environment Entity service variable.
   * @param \Drupal\acsf_module_listing\Services\AcsfApiV1Service $acsfApiV1Service
   *   ACSF API V1 service variable.
   * @param \Drupal\acsf_module_listing\Services\SshShellService $shellService
   *   SSH Shell Service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, CacheBackendInterface $cache, RequestStack $requestStack, ModuleExtensionList $moduleList, AcsfEnvironmentEntityService $acsfEnvironmentEntityService, AcsfApiV1Service $acsfApiV1Service, SshShellService $shellService) {
    $this->config = $config_factory->get('acsf_module_listing.modules_usage_configuration');
    $this->cache = $cache;
    $this->moduleList = $moduleList;
    $this->requestStack = $requestStack;
    $this->batchBuilder = new BatchBuilder();
    $this->acsfEnvironmentEntityService = $acsfEnvironmentEntityService;
    $this->acsfApiV1Service = $acsfApiV1Service;
    $this->sshShellService = $shellService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('cache.default'),
      $container->get('request_stack'),
      $container->get('extension.list.module'),
      $container->get('acsf_module_listing.acsf_environment_entity'),
      $container->get('acsf_module_listing.acsf_api_v1_service'),
      $container->get('acsf_module_listing.ssh_service'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'acsf_module_listing_modules_usage_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    if (function_exists('ssh2_connect')) {
      $values = $this->config->get('configuration');

      $acsf_values = [];
      $acsf_entities = \Drupal::entityTypeManager()
        ->getStorage('acsf_environment_entity')
        ->loadMultiple();

      foreach ($acsf_entities as $acsf_entity) {
        $acsf_values[$acsf_entity->id()] = $acsf_entity->label();
      }

      $form['server'] = [
        '#type' => 'select',
        '#title' => $this->t('Select server to search'),
        '#options' => $acsf_values,
        '#required' => TRUE,
      ];

      $form['module_search'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Module name:'),
        '#required' => TRUE,
        '#placeholder' => $this->t('Please, enter machine name or module title'),
      ];

      $form['ignore_cache'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Ignore cached results.'),
        '#description' => $this->t('Check this if you want to receive actual data. If you already did a request, the results will be saved for @seconds seconds. This means, that your next query will be processed with cached data and these results might be outdated.', ['@seconds' => $values['cache_lifetime']]),
      ];

      $form['actions'] = ['#type' => 'actions', '#weight' => 2];
      $form['actions']['run'] = [
        '#type' => 'submit',
        '#value' => $this->t('Run search'),
      ];

      if ($cache = $this->requestStack->getCurrentRequest()->get('result')) {
        $cached_modules = $this->cache->get($cache);

        $rows = [];
        $header = [$this->t('Website'), $this->t('Modules')];

        if (!empty($cached_modules->data)) {
          foreach ($cached_modules->data as $website => $cached_modules_array) {
            $count = count($cached_modules_array);
            if ($count) {
              $cached_module = array_shift($cached_modules_array);

              $cached_module = $this->formatModuleName($cached_module);
              $rows[] = [
                ['data' => $website, 'rowspan' => $count],
                ['data' => $cached_module['title'] . '(' . $cached_module['version'] . ')'],
              ];
              foreach ($cached_modules_array as $cached_module) {
                $cached_module = $this->formatModuleName($cached_module);
                $rows[] = [['data' => $cached_module['title']]];
              }
            }
          }
        }

        $form['results'] = [
          '#weight' => 3,
          '#type' => 'table',
          '#header' => $header,
          '#rows' => $rows,
          '#empty' => $this->t('No results found'),
        ];
      }
    }
    else {
      $url = Url::fromUri('https://www.php.net/manual/en/book.ssh2.php');
      $url->setOptions(['attributes' => ['target' => '_blank']]);

      $form['results'] = [
        '#weight' => 3,
        '#type' => 'inline_template',
        '#template' => $this->t(
          'Please, enable and configure @link before starting search.',
          [
            '@link' => Link::fromTextAndUrl('SSH2 library', $url)
              ->toString(),
          ]
        ),
      ];
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $server_id = $form_state->getValue('server');
    $acsf_environment = $this->acsfEnvironmentEntityService->loadAcsfEnvironmentById($server_id);
    $key = 'acsf_module_listing_search.' . Html::cleanCssIdentifier($form_state->getValue('module_search'));
    $cache = $this->cache->get($key);
    $ignore_cache = $form_state->getValue('ignore_cache');

    if (isset($cache->data) && !$ignore_cache) {
      return new RedirectResponse('/admin/config/services/modules-usage/search?result=' . $key);
    }

    if ($acsf_environment) {
      $response = $this->acsfApiV1Service->getWebsitesList($acsf_environment);

      if ($response && !empty($response['sites'])) {
        $this->batchBuilder
          ->setTitle($this->t('Processing'))
          ->setInitMessage($this->t('Initializing.'))
          ->setProgressMessage($this->t('Completed @current of @total.'))
          ->setErrorMessage($this->t('An error has occurred.'));

        $this->batchBuilder->setFile(
          $this->moduleList->getPath('acsf_module_listing')
          . '/src/Form/ACSFModulesListingForm.php'
        );

        foreach ($response['sites'] as $num => $website) {
          $this->batchBuilder->addOperation(
            [$this, 'processItem'], [
              [
                'website' => $website['domain'],
                'current' => ++$num,
                'total' => $response['count'],
                'acsf_environment' => $acsf_environment,
                'input' => $form_state->getValue('module_search'),
                'ignore_cache' => $ignore_cache,
              ],
            ]
          );
        }

        $this->batchBuilder->setFinishCallback([$this, 'finished']);
        batch_set($this->batchBuilder->toArray());
      }
      else {
        $this->messenger()
          ->addError('Batch process has been failed. Please, check logs.');
      }
    }
  }

  /**
   * Batch process item function.
   *
   * @param array $data
   *   Data variable.
   * @param array $context
   *   Batch context variable.
   */
  public function processItem(array $data, array &$context) {
    $context['message'] = $this->t(
      'Now processing website: ":website"',
      [
        ':website' => $data['website'],
      ]
    );

    $context['results']['processed'] = $data['current'];
    $context['results']['modules'][$data['website']] = $this->sshShellService->getWebsiteModulesList($data['acsf_environment'], $data['website'], $data['input'], $data['ignore_cache']);
    $context['results']['input'] = $data['input'];
  }

  /**
   * Batch finish function.
   *
   * @param bool $success
   *   Status variable.
   * @param array $results
   *   Results data variable.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   Redirect response.
   */
  public function finished(bool $success, array $results) {
    $this->messenger()->addStatus(
      $this->t(
        'Number of websites affected by batch: @count', [
          '@count' => $results['processed'] ?? 0,
        ]
      )
    );
    if ($success && isset($results['input'])) {
      $key = 'acsf_module_listing_search.' . Html::cleanCssIdentifier($results['input']);
      $this->cache->set($key, $results['modules'], time() + $this->config->get('configuration.cache_lifetime'));

      return new RedirectResponse('/admin/config/services/modules-usage/search?result=' . $key);
    }
    $this->messenger()->addError('Batch process has been failed');
  }

  /**
   * Array formatter for drush response.
   *
   * @param array $module
   *   Module data array.
   *
   * @return array
   *   Formatted data array.
   */
  protected function formatModuleName(array $module) {
    $formatted = [
      'version' => array_pop($module),
      'status' => array_pop($module),
      'machine_name' => array_pop($module),
      'package' => array_shift($module),
    ];
    $formatted['title'] = implode(" ", $module);

    return $formatted;

  }

}
